// Shared user store for all API routes
// Uses a JSON file for persistence across server restarts

import * as fs from 'fs';
import * as path from 'path';

export interface User {
  id: string;
  username: string;
  email: string;
  password: string;
  role: 'admin' | 'editor' | 'viewer';
  createdAt: string;
}

// Create users data directory if it doesn't exist
const dataDir = path.join(process.cwd(), 'data');
if (!fs.existsSync(dataDir)) {
  fs.mkdirSync(dataDir, { recursive: true });
}

const usersFilePath = path.join(dataDir, 'users.json');

// Initialize with default admin user if file doesn't exist
const initializeUsersFile = () => {
  if (!fs.existsSync(usersFilePath)) {
    const defaultUsers: User[] = [
      {
        id: '1',
        username: 'Admin',
        email: 'admin@busynepal.com',
        password: 'Busynepal@34143',
        role: 'admin',
        createdAt: new Date().toISOString(),
      },
    ];
    fs.writeFileSync(usersFilePath, JSON.stringify(defaultUsers, null, 2));
  }
};

// Load users from file
const loadUsers = (): User[] => {
  try {
    initializeUsersFile();
    const data = fs.readFileSync(usersFilePath, 'utf-8');
    return JSON.parse(data);
  } catch (error) {
    console.error('Error loading users:', error);
    initializeUsersFile();
    return JSON.parse(fs.readFileSync(usersFilePath, 'utf-8'));
  }
};

// Save users to file
const saveUsers = (users: User[]) => {
  try {
    fs.writeFileSync(usersFilePath, JSON.stringify(users, null, 2));
  } catch (error) {
    console.error('Error saving users:', error);
  }
};

export const userStore = {
  getAll(): User[] {
    return loadUsers();
  },

  findById(id: string): User | undefined {
    return this.getAll().find((u) => u.id === id);
  },

  findByUsername(username: string): User | undefined {
    return this.getAll().find((u) => u.username === username);
  },

  findByEmail(email: string): User | undefined {
    return this.getAll().find((u) => u.email === email);
  },

  create(user: Omit<User, 'id' | 'createdAt'>): User {
    const users = this.getAll();
    const newUser: User = {
      ...user,
      id: Date.now().toString(),
      createdAt: new Date().toISOString(),
    };
    users.push(newUser);
    saveUsers(users);
    return newUser;
  },

  update(id: string, updates: Partial<User>): User | undefined {
    const users = this.getAll();
    const index = users.findIndex((u) => u.id === id);
    if (index === -1) return undefined;

    users[index] = {
      ...users[index],
      ...updates,
      id: users[index].id, // Never update ID
      createdAt: users[index].createdAt, // Never update createdAt
    };
    saveUsers(users);
    return users[index];
  },

  delete(id: string): boolean {
    const users = this.getAll();
    const index = users.findIndex((u) => u.id === id);
    if (index === -1) return false;

    users.splice(index, 1);
    saveUsers(users);
    return true;
  },

  exists(username: string, email: string, excludeId?: string): boolean {
    return this.getAll().some(
      (u) =>
        (u.username === username || u.email === email) &&
        u.id !== excludeId
    );
  },
};
