'use client';

import { useState, useEffect } from 'react';
import Link from 'next/link';
import Image from 'next/image'; 
import { Button } from '@/components/ui/button';
import { Menu, X, Phone, Mail, Facebook, Youtube, Linkedin, Instagram, ChevronDown } from 'lucide-react';
import { EmailLink, PhoneLink } from './protected-contact';

const Header = () => {
  const [isScrolled, setIsScrolled] = useState(false);
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false);
  const [productsDropdownOpen, setProductsDropdownOpen] = useState(false);

  useEffect(() => {
    const handleScroll = () => {
      setIsScrolled(window.scrollY > 50);
    };

    window.addEventListener('scroll', handleScroll);
    return () => window.removeEventListener('scroll', handleScroll);
  }, []);

  const navigation = [
    { name: 'Home', href: '/' },
    { name: 'Products', href: '#', dropdown: true },
    { name: 'Services', href: '/services' },
    { name: 'Blog', href: '/blog' },
    { name: 'About', href: '/about' },
    { name: 'Downloads', href: '/downloads' },
    { name: 'Contact', href: '/contact' },
  ];

  const products = [
    { name: 'Busy 21 Basic', color: 'Blue', href: '/products/busy-21-basic' },
    { name: 'Busy 21 Standard', color: 'Saffron', href: '/products/busy-21-standard' },
    { name: 'Busy 21 Enterprise', color: 'Emerald', href: '/products/busy-21-enterprise' },
    { name: 'Busy On Cloud', color: 'Cloud', href: '/products/busy-on-cloud' },
    { name: 'Tally Prime', color: 'Prime', href: '/products/tally-prime' },
    { name: 'Tally On Cloud', color: 'Cloud', href: '/products/tally-on-cloud' },
  ];

  return (
    <header className="fixed top-0 left-0 right-0 z-50 transition-all duration-300">
      {/* Top Bar */}
      <div className="gradient-primary text-white py-2 hidden md:block">
        <div className="container-custom">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-6">
              <PhoneLink number="+9779857834143" label="+977-9857834143" />
              <EmailLink address="info@busynepal.com" label="Contact via Email" className="flex items-center gap-2 text-sm hover:underline transition-colors" />
              <a href="https://busynepal.com" target="_blank" rel="noopener noreferrer" className="text-sm hover:underline transition-colors">
                www.busynepal.com
              </a>
            </div>
            <div className="flex items-center gap-4">
              <a
                href="https://www.facebook.com/profile.php?id=61574548711911"
                target="_blank"
                rel="noopener noreferrer"
                className="hover:opacity-80 transition-opacity"
                aria-label="Facebook"
              >
                <Facebook className="w-4 h-4" />
              </a>
              <a
                href="https://youtube.com/@techhbijay"
                target="_blank"
                rel="noopener noreferrer"
                className="hover:opacity-80 transition-opacity"
                aria-label="YouTube"
              >
                <Youtube className="w-4 h-4" />
              </a>
              <a
                href="https://www.linkedin.com/in/busy-software-nepal-8419453a8/"
                target="_blank"
                rel="noopener noreferrer"
                className="hover:opacity-80 transition-opacity"
                aria-label="LinkedIn"
              >
                <Linkedin className="w-4 h-4" />
              </a>
              <a
                href="https://instagram.com/nehainfotechh"
                target="_blank"
                rel="noopener noreferrer"
                className="hover:opacity-80 transition-opacity"
                aria-label="Instagram"
              >
                <Instagram className="w-4 h-4" />
              </a>
            </div>
          </div>
        </div>
      </div>

      {/* Main Navigation */}
      <nav
        className={`bg-white shadow-sm transition-all duration-300 ${
          isScrolled ? 'py-2' : 'py-4'
        } ${mobileMenuOpen ? 'mobile-open' : ''}`}
      >
        <div className="container-custom">
          <div className="flex items-center justify-between">
            {/* Logo */}
            <Link href="/" className="flex items-center gap-3 group">
              <Image
                src="/logo.png"
                alt="BusyNepal Logo"
                width={50}
                height={50}
                className="rounded-lg"
                priority
              />
              <div>
                <h2 className="text-xl font-bold text-[#003d77] group-hover:text-[#005a9f] transition-colors">
                  Neha Infotech
                </h2>
                <p className="text-xs text-gray-500">Authorized Distributor of Busy in Nepal</p>
              </div>
            </Link>

            {/* Desktop Navigation */}
            <div className="hidden lg:flex items-center gap-8">
              {navigation.map((item) => (
                <div key={item.name} className="relative group">
                  {item.dropdown ? (
                    <>
                      <button className="nav-link text-sm font-medium text-gray-700 hover:text-[#005a9f] transition-colors flex items-center gap-1">
                        {item.name}
                        <ChevronDown className="w-4 h-4" />
                      </button>
                      {/* Desktop Dropdown */}
                      <div className="absolute left-0 mt-0 w-56 bg-white shadow-lg rounded-lg opacity-0 invisible group-hover:opacity-100 group-hover:visible transition-all duration-300 py-2 z-50">
                        {products.map((product) => (
                          <Link
                            key={product.name}
                            href={product.href}
                            className="block px-4 py-3 text-sm text-gray-700 hover:bg-blue-50 hover:text-[#005a9f] transition-colors border-b border-gray-100 last:border-0"
                          >
                            <div className="font-medium">{product.name}</div>
                            <div className="text-xs text-gray-500">({product.color})</div>
                          </Link>
                        ))}
                      </div>
                    </>
                  ) : (
                    <Link
                      href={item.href}
                      className="nav-link text-sm font-medium text-gray-700 hover:text-[#005a9f] transition-colors"
                    >
                      {item.name}
                    </Link>
                  )}
                </div>
              ))}
            </div>

            {/* Call Now Button - Desktop */}
            <div className="hidden lg:flex items-center gap-4">
              <a href="tel:+9779857834143">
                <Button className="bg-[#e05a5a] hover:bg-[#f37474] text-white rounded-lg px-6 py-6 font-medium transition-colors">
                  <Phone className="w-4 h-4 mr-2" />
                  Call Now
                </Button>
              </a>
            </div>

            {/* Mobile Menu Button */}
            <button
              onClick={() => setMobileMenuOpen(!mobileMenuOpen)}
              className="lg:hidden p-2 hover:bg-gray-100 rounded-lg transition-colors"
              aria-label="Toggle menu"
            >
              {mobileMenuOpen ? (
                <X className="w-6 h-6" />
              ) : (
                <Menu className="w-6 h-6" />
              )}
            </button>
          </div>

          {/* Mobile Menu */}
          {mobileMenuOpen && (
            <div className="lg:hidden mt-4 pt-4 border-t border-gray-200 animate-fade-in">
              <div className="flex flex-col gap-4">
                {navigation.map((item) => (
                  <div key={item.name}>
                    {item.dropdown ? (
                      <>
                        <button
                          onClick={() => setProductsDropdownOpen(!productsDropdownOpen)}
                          className="w-full text-left text-base font-medium text-gray-700 hover:text-[#005a9f] transition-colors py-2 border-b border-gray-100 flex items-center justify-between"
                        >
                          {item.name}
                          <ChevronDown className={`w-4 h-4 transition-transform ${productsDropdownOpen ? 'rotate-180' : ''}`} />
                        </button>
                        {/* Mobile Dropdown */}
                        {productsDropdownOpen && (
                          <div className="bg-blue-50 border-l-2 border-[#005a9f] animate-fade-in">
                            {products.map((product) => (
                              <Link
                                key={product.name}
                                href={product.href}
                                onClick={() => {
                                  setMobileMenuOpen(false);
                                  setProductsDropdownOpen(false);
                                }}
                                className="block px-4 py-3 text-sm text-gray-700 hover:text-[#005a9f] transition-colors border-b border-gray-200 last:border-0"
                              >
                                <div className="font-medium">{product.name}</div>
                                <div className="text-xs text-gray-500">({product.color})</div>
                              </Link>
                            ))}
                          </div>
                        )}
                      </>
                    ) : (
                      <Link
                        href={item.href}
                        onClick={() => setMobileMenuOpen(false)}
                        className="text-base font-medium text-gray-700 hover:text-[#005a9f] transition-colors py-2 border-b border-gray-100"
                      >
                        {item.name}
                      </Link>
                    )}
                  </div>
                ))}

                {/* Mobile Top Bar Info */}
                <div className="pt-4 mt-2 border-t border-gray-200">
                  <div className="flex flex-col gap-3 text-sm">
                    <a
                      href="tel:+9779857834143"
                      className="flex items-center gap-2 text-gray-600 hover:text-[#005a9f] transition-colors"
                    >
                      <Phone className="w-4 h-4" />
                      +977-9857834143
                    </a>
                    <a
                      href="mailto:info@busynepal.com"
                      className="flex items-center gap-2 text-gray-600 hover:text-[#005a9f] transition-colors"
                    >
                      <Mail className="w-4 h-4" />
                      info@busynepal.com
                    </a>
                    <div className="flex items-center gap-4 pt-2">
                      <a
                        href="https://www.facebook.com/profile.php?id=61574548711911"
                        target="_blank"
                        rel="noopener noreferrer"
                        className="text-gray-600 hover:text-[#005a9f] transition-colors"
                        aria-label="Facebook"
                      >
                        <Facebook className="w-5 h-5" />
                      </a>
                      <a
                        href="https://youtube.com/@techhbijay"
                        target="_blank"
                        rel="noopener noreferrer"
                        className="text-gray-600 hover:text-[#005a9f] transition-colors"
                        aria-label="YouTube"
                      >
                        <Youtube className="w-5 h-5" />
                      </a>
                      <a
                        href="https://www.linkedin.com/in/busy-software-nepal-8419453a8/"
                        target="_blank"
                        rel="noopener noreferrer"
                        className="text-gray-600 hover:text-[#005a9f] transition-colors"
                        aria-label="LinkedIn"
                      >
                        <Linkedin className="w-5 h-5" />
                      </a>
                      <a
                        href="https://www.instagram.com/nehainfotechh/"
                        target="_blank"
                        rel="noopener noreferrer"
                        className="text-gray-600 hover:text-[#005a9f] transition-colors"
                        aria-label="Instagram"
                      >
                        <Instagram className="w-5 h-5" />
                      </a>
                    </div>
                  </div>

                  <a href="tel:+9779857834143" className="pt-4">
                    <Button className="w-full bg-[#e05a5a] hover:bg-[#f37474] text-white rounded-lg py-6 font-medium transition-colors">
                      <Phone className="w-4 h-4 mr-2" />
                      Call Now
                    </Button>
                  </a>
                </div>
              </div>
            </div>
          )}
        </div>
        {/* Promotional Bar */}
<div className="bg-[#e05a5a] text-white py-3 text-center">
  <div className="container-custom">
    <p className="text-sm md:text-base font-semibold">
      🎉 BUY TODAY AND RENEW AFTER 3 YEARS
    </p>
  </div>
</div>
      </nav>
    </header>
  );
};

export default Header;
