'use client';

import { useState } from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Eye, Edit, Trash2, Plus, Loader2 } from 'lucide-react';

interface Page {
  id: string;
  title: string;
  slug: string;
  content: string;
  createdAt: string;
  editorType: 'visual' | 'code';
}

interface PagesTabProps {
  pages: Page[];
  loading: boolean;
  onCreateNew: () => void;
  onEdit: (page: Page) => void;
  onDelete: (pageId: string) => void;
  onView: (page: Page) => void;
}

export const PagesTab = ({
  pages,
  loading,
  onCreateNew,
  onEdit,
  onDelete,
  onView,
}: PagesTabProps) => {
  const [selectedPage, setSelectedPage] = useState<Page | null>(null);
  const [deleting, setDeleting] = useState<string | null>(null);

  const handleDelete = async (pageId: string) => {
    if (!confirm('Are you sure you want to delete this page?')) return;

    setDeleting(pageId);
    try {
      await onDelete(pageId);
      if (selectedPage?.id === pageId) {
        setSelectedPage(null);
      }
    } finally {
      setDeleting(null);
    }
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex justify-between items-center">
        <div>
          <h2 className="text-2xl font-bold">Pages Management</h2>
          <p className="text-gray-600 text-sm">Create and manage pages on your website</p>
        </div>
        <Button onClick={onCreateNew} className="gap-2">
          <Plus className="w-4 h-4" />
          New Page
        </Button>
      </div>

      {/* Content Grid */}
      <div className="grid lg:grid-cols-3 gap-8">
        {/* Pages List */}
        <div className="lg:col-span-2">
          <Card>
            <CardHeader>
              <CardTitle>Available Pages</CardTitle>
              <CardDescription>Click on a page to view details and manage it</CardDescription>
            </CardHeader>
            <CardContent>
              {loading ? (
                <div className="flex items-center justify-center py-8">
                  <Loader2 className="w-6 h-6 animate-spin text-gray-400" />
                </div>
              ) : pages.length === 0 ? (
                <div className="text-center py-8">
                  <p className="text-gray-500 mb-4">No pages yet</p>
                  <Button onClick={onCreateNew} size="sm">
                    Create Your First Page
                  </Button>
                </div>
              ) : (
                <div className="space-y-3">
                  {pages.map((page) => (
                    <div
                      key={page.id}
                      onClick={() => setSelectedPage(page)}
                      className={`p-4 border rounded-lg cursor-pointer transition-all ${
                        selectedPage?.id === page.id
                          ? 'border-blue-500 bg-blue-50'
                          : 'border-gray-200 hover:border-gray-300'
                      }`}
                    >
                      <div className="flex items-start justify-between">
                        <div className="flex-1">
                          <h3 className="font-semibold">{page.title}</h3>
                          <p className="text-sm text-gray-600">/{page.slug}</p>
                          <div className="flex items-center gap-2 mt-2">
                            <Badge variant="outline" className="text-xs">
                              {page.editorType === 'visual' ? '🎨 Visual' : '💻 Code'}
                            </Badge>
                            <span className="text-xs text-gray-500">
                              {new Date(page.createdAt).toLocaleDateString()}
                            </span>
                          </div>
                        </div>
                        <Eye className="w-4 h-4 text-gray-400 flex-shrink-0" />
                      </div>
                    </div>
                  ))}
                </div>
              )}
            </CardContent>
          </Card>
        </div>

        {/* Details Panel */}
        <div>
          {selectedPage ? (
            <Card>
              <CardHeader>
                <CardTitle className="text-lg">Page Details</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <label className="text-xs font-semibold text-gray-500 uppercase">Title</label>
                  <p className="text-base font-medium">{selectedPage.title}</p>
                </div>

                <div>
                  <label className="text-xs font-semibold text-gray-500 uppercase">URL Slug</label>
                  <p className="text-sm text-gray-700 font-mono">/{selectedPage.slug}</p>
                </div>

                <div>
                  <label className="text-xs font-semibold text-gray-500 uppercase">Editor Type</label>
                  <p className="text-sm">
                    <Badge variant="outline">
                      {selectedPage.editorType === 'visual' ? '🎨 Visual Editor' : '💻 Code Editor'}
                    </Badge>
                  </p>
                </div>

                <div>
                  <label className="text-xs font-semibold text-gray-500 uppercase">Created</label>
                  <p className="text-sm text-gray-600">
                    {new Date(selectedPage.createdAt).toLocaleString()}
                  </p>
                </div>

                <div className="flex flex-col gap-2 pt-4 border-t">
                  <Button
                    onClick={() => onEdit(selectedPage)}
                    variant="outline"
                    className="w-full gap-2"
                  >
                    <Edit className="w-4 h-4" />
                    Edit Page
                  </Button>
                  <Button
                    onClick={() => handleDelete(selectedPage.id)}
                    disabled={deleting === selectedPage.id}
                    variant="destructive"
                    className="w-full"
                  >
                    {deleting === selectedPage.id ? (
                      <>
                        <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                        Deleting...
                      </>
                    ) : (
                      <>
                        <Trash2 className="w-4 h-4 mr-2" />
                        Delete Page
                      </>
                    )}
                  </Button>
                </div>
              </CardContent>
            </Card>
          ) : (
            <Card>
              <CardContent className="py-12">
                <p className="text-center text-gray-500">
                  Select a page to view details
                </p>
              </CardContent>
            </Card>
          )}
        </div>
      </div>
    </div>
  );
};
