'use client';

import { useState } from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { X, Loader2, Code, Layout } from 'lucide-react';

interface BlogPostModalProps {
  isOpen: boolean;
  onClose: () => void;
  onCreate: (postData: {
    title: string;
    slug: string;
    excerpt: string;
    content: string;
    category: string;
    editorType: 'visual' | 'code';
  }) => void;
  editData?: {
    id: string;
    title: string;
    slug: string;
    excerpt: string;
    content: string;
    category: string;
    editorType: 'visual' | 'code';
  } | null;
}

const BLOG_CATEGORIES = ['Accounting', 'VAT Compliance', 'Inventory', 'Reports', 'Migration', 'Comparison'];

export const BlogPostModal = ({ isOpen, onClose, onCreate, editData }: BlogPostModalProps) => {
  const [title, setTitle] = useState(editData?.title || '');
  const [slug, setSlug] = useState(editData?.slug || '');
  const [excerpt, setExcerpt] = useState(editData?.excerpt || '');
  const [content, setContent] = useState(editData?.content || '');
  const [category, setCategory] = useState(editData?.category || 'Accounting');
  const [editorType, setEditorType] = useState<'visual' | 'code'>(editData?.editorType || 'visual');
  const [isSubmitting, setIsSubmitting] = useState(false);

  const handleTitleChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const value = e.target.value;
    setTitle(value);
    setSlug(value.toLowerCase().replace(/\s+/g, '-'));
  };

  const handleSubmit = async () => {
    if (!title.trim() || !slug.trim() || !excerpt.trim() || !content.trim() || !category) {
      alert('Please fill in all fields');
      return;
    }

    setIsSubmitting(true);
    try {
      onCreate({
        title,
        slug,
        excerpt,
        content,
        category,
        editorType,
      });
      if (!editData) {
        setTitle('');
        setSlug('');
        setExcerpt('');
        setContent('');
        setCategory('Accounting');
      }
      onClose();
    } finally {
      setIsSubmitting(false);
    }
  };

  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center p-4">
      <Card className="w-full max-w-3xl max-h-[90vh] overflow-y-auto">
        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-4">
          <div>
            <CardTitle>{editData ? 'Edit Blog Post' : 'Create New Blog Post'}</CardTitle>
            <CardDescription>
              {editData ? 'Update your blog post' : 'Add a new article to your blog'}
            </CardDescription>
          </div>
          <button
            onClick={onClose}
            className="text-gray-500 hover:text-gray-700"
            title="Close modal"
            aria-label="Close modal"
          >
            <X className="w-5 h-5" />
          </button>
        </CardHeader>

        <CardContent className="space-y-6">
          {/* Post Info */}
          <div className="space-y-4">
            <div>
              <label className="block text-sm font-medium mb-2">Post Title</label>
              <input
                type="text"
                value={title}
                onChange={handleTitleChange}
                placeholder="Enter post title"
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
              />
            </div>

            <div className="grid grid-cols-2 gap-4">
              <div>
                <label className="block text-sm font-medium mb-2">URL Slug</label>
                <input
                  type="text"
                  value={slug}
                  onChange={(e) => setSlug(e.target.value)}
                  placeholder="post-url-slug"
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                />
              </div>

              <div>
                <label htmlFor="category" className="block text-sm font-medium mb-2">Category</label>
                <select
                  id="category"
                  value={category}
                  onChange={(e) => setCategory(e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                >
                  {BLOG_CATEGORIES.map((cat) => (
                    <option key={cat} value={cat}>
                      {cat}
                    </option>
                  ))}
                </select>
              </div>
            </div>

            <div>
              <label className="block text-sm font-medium mb-2">Excerpt</label>
              <textarea
                value={excerpt}
                onChange={(e) => setExcerpt(e.target.value)}
                placeholder="Brief summary of the post (shown in listings)"
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 min-h-[80px]"
              />
            </div>
          </div>

          {/* Editor Tabs */}
          <Tabs value={editorType} onValueChange={(value) => setEditorType(value as 'visual' | 'code')}>
            <TabsList className="grid w-full grid-cols-2">
              <TabsTrigger value="visual" className="gap-2">
                <Layout className="w-4 h-4" />
                Visual Editor
              </TabsTrigger>
              <TabsTrigger value="code" className="gap-2">
                <Code className="w-4 h-4" />
                Code Editor
              </TabsTrigger>
            </TabsList>

            <TabsContent value="visual" className="space-y-4">
              <div>
                <label className="block text-sm font-medium mb-2">Post Content</label>
                <textarea
                  value={content}
                  onChange={(e) => setContent(e.target.value)}
                  placeholder="Enter your blog post content here..."
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 min-h-[300px] font-sans"
                />
                <p className="text-xs text-gray-500 mt-2">Supports plain text, markdown, and basic HTML</p>
              </div>
            </TabsContent>

            <TabsContent value="code" className="space-y-4">
              <div>
                <label className="block text-sm font-medium mb-2">Post Code (JSX/TSX)</label>
                <textarea
                  value={content}
                  onChange={(e) => setContent(e.target.value)}
                  placeholder="Enter your JSX/TSX code here..."
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 min-h-[300px] font-mono text-sm"
                />
                <p className="text-xs text-gray-500 mt-2">Write valid React/JSX code</p>
              </div>
            </TabsContent>
          </Tabs>

          {/* Actions */}
          <div className="flex gap-3 justify-end pt-4 border-t">
            <Button variant="outline" onClick={onClose}>
              Cancel
            </Button>
            <Button onClick={handleSubmit} disabled={isSubmitting}>
              {isSubmitting && <Loader2 className="w-4 h-4 mr-2 animate-spin" />}
              {editData ? 'Update Post' : 'Create Post'}
            </Button>
          </div>
        </CardContent>
      </Card>
    </div>
  );
};
