import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardFooter, CardTitle, CardDescription } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import Image from 'next/image';
import {
  CheckCircle2,
  BarChart3,
  Download,
  Users,
  Shield,
  Zap,
  FileText,
  Database,
  TrendingUp,
  MousePointerClick,
  Globe,
  Lock,
  Printer,
  Check,
  Server,
  HardDrive,
  Settings,
  LayoutGrid,
} from 'lucide-react';
import Link from 'next/link';

export default function TallyPrimePage() {
  const features = [
    { icon: Zap, title: 'Amazing Speed', description: 'Experience report generation at lightning speed with zero latency.' },
    { icon: MousePointerClick, title: 'Simplified Navigation', description: 'Intuitive interface that helps you discover features and reports easily.' },
    { icon: Globe, title: 'Go Remote', description: 'Access your business data from anywhere, anytime with secure connectivity.' },
    { icon: Shield, title: 'Secure Data', description: 'Bank-grade security with advanced access controls and data encryption.' },
    { icon: LayoutGrid, title: 'Multi-Tasking', description: 'Handle multiple tasks simultaneously without switching screens.' },
    { icon: Printer, title: 'Instant Reports', description: 'Generate and print invoices, receipts, and reports in a single click.' },
  ];

  const capabilities = [
    {
      title: 'Accounting',
      items: ['GST Compliant', 'Multi-Currency', 'Cost Centers', 'Budgets & Controls', 'Interest Calculation'],
    },
    {
      title: 'Inventory',
      items: ['Stock Summary', 'Batch/Serial Tracking', 'Multiple Godowns', 'Manufacturing (Bill of Materials)', 'Order Processing'],
    },
    {
      title: 'Reports & Analytics',
      items: ['Ratio Analysis', 'Cash Flow', 'Funds Flow', 'Balance Sheet', 'Comparative Analysis'],
    },
    {
      title: 'Payroll & HR',
      items: ['Employee Groups', 'Attendance', 'Payroll Auto-Fill', 'Income Tax', 'PF/ESI Integration'],
    },
  ];

  const plans = [
    {
      name: 'TallyPrime Silver',
      price: 'NPR 50,000',
      description: 'For a single user. Ideal for small businesses and proprietors.',
      highlight: false,
      users: 'Single User',
      features: [
        'Single User License',
        'Lifetime Validity',
        'Basic Accounting & Inventory',
        'GST Reports & Filing',
        'Remote Access Support',
        'Standard Support',
      ],
    },
    {
      name: 'TallyPrime Gold',
      price: 'NPR 150,000',
      description: 'For multiple users. Best for growing teams requiring collaboration.',
      highlight: true,
      badge: 'Most Popular',
      users: 'Unlimited Users (LAN)',
      features: [
        'Unlimited Concurrent Users',
        'Lifetime Validity',
        'Advanced Accounting & Inventory',
        'Branch Consolidation',
        'Data Synchronization',
        'Priority Support',
      ],
    },
  ];

  const comparisonData = [
    { feature: 'User Capacity', silver: '1 User', gold: 'Unlimited (LAN)' },
    { feature: 'Remote Access', silver: 'Yes (via Tally.NET)', gold: 'Yes (via Tally.NET)' },
    { feature: 'Data Synchronization', silver: 'No', gold: 'Yes' },
    { feature: 'Audit Logging / Admin', silver: 'Basic', gold: 'Standard' },
    { feature: 'Business Reporting', silver: 'Standard', gold: 'Advanced' },
    { feature: 'Manufacturing Capabilities', silver: 'Basic', gold: 'Full' },
    { feature: 'Support Level', silver: 'Email', gold: 'Phone & Email' },
    { feature: 'Ideal For', silver: 'Single PC', gold: 'Office Network' },
  ];

  return (
    <>
      {/* Hero Section */}
      <section className="bg-gradient-to-br from-cyan-500 to-blue-600 text-white py-20 pt-40">
        <div className="container mx-auto px-4">
          <div className="grid lg:grid-cols-2 gap-8 items-center">
            {/* Content */}
            <div>
              <Badge className="bg-white/20 hover:bg-white/20 text-white border-white/30 mb-6">
                Business Management Software
              </Badge>
              <h1 className="text-4xl lg:text-6xl font-bold mb-6 tracking-tight">
                TallyPrime
              </h1>
              <p className="text-xl text-cyan-50 mb-8 leading-relaxed">
                Simplify your business. A product that evolves with your needs, designed for speed, reliability, and simplicity.
              </p>
              <div className="flex flex-wrap gap-4">
                <Link href="/downloads">
                  <Button size="lg" className="bg-white text-cyan-700 hover:bg-cyan-50 font-semibold px-8">
                    <Download className="mr-2 h-5 w-5" />
                    Download Free Trial
                  </Button>
                </Link>
                <Link href="/contact">
                  <Button size="lg" variant="outline" className="border-white text-white hover:bg-white hover:text-cyan-700 px-8">
                    Buy Now
                  </Button>
                </Link>
              </div>
            </div>
            
            {/* Image */}
            <div className="flex items-center justify-center">
              <div className="relative w-full max-w-md aspect-square bg-white/10 rounded-2xl border-2 border-white/20 overflow-hidden">
                <Image
                  src="/TallyPrime.png"
                  alt="TallyPrime Product"
                  fill
                  className="object-cover"
                  priority
                />
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Features */}
      <section className="py-16 bg-white">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Why Choose TallyPrime?
            </h2>
            <p className="text-lg text-gray-600 max-w-2xl mx-auto">
              Features designed to make your business life easier.
            </p>
          </div>
          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
            {features.map((feature, index) => (
              <Card key={index} className="border-2 border-transparent hover:border-cyan-200 transition-all hover:shadow-lg">
                <CardContent className="p-6">
                  <div className="w-14 h-14 bg-cyan-100 rounded-2xl flex items-center justify-center mb-5">
                    <feature.icon className="h-7 w-7 text-cyan-600" />
                  </div>
                  <h3 className="text-xl font-bold text-gray-900 mb-3">
                    {feature.title}
                  </h3>
                  <p className="text-gray-600 leading-relaxed">
                    {feature.description}
                  </p>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Capabilities */}
      <section className="py-16 bg-gray-50 border-y border-gray-100">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Manage Your Business End-to-End
            </h2>
          </div>
          <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-6">
            {capabilities.map((capability, index) => (
              <Card key={index} className="border-none shadow-sm bg-white h-full">
                <CardHeader>
                  <h3 className="text-lg font-bold text-gray-900">
                    {capability.title}
                  </h3>
                </CardHeader>
                <CardContent>
                  <ul className="space-y-3">
                    {capability.items.map((item, i) => (
                      <li key={i} className="flex items-center gap-3 text-sm text-gray-600">
                        <CheckCircle2 className="h-4 w-4 text-cyan-500 flex-shrink-0" />
                        {item}
                      </li>
                    ))}
                  </ul>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Pricing Section */}
      <section id="pricing" className="py-20 bg-white">
        <div className="container mx-auto px-4">
          <div className="text-center mb-16">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Flexible Licensing
            </h2>
            <p className="text-lg text-gray-600">
              Choose the edition that fits your business size.
            </p>
          </div>

          <div className="grid lg:grid-cols-2 gap-8 max-w-4xl mx-auto items-start">
            {plans.map((plan, index) => (
              <Card 
                key={index} 
                className={`relative flex flex-col ${plan.highlight ? 'border-cyan-500 shadow-2xl scale-105 z-10' : 'border-gray-200 shadow-sm'}`}
              >
                {plan.highlight && (
                  <div className="absolute top-0 right-0 bg-cyan-500 text-white text-xs font-bold px-3 py-1 rounded-bl-lg">
                    {plan.badge}
                  </div>
                )}
                <CardHeader className="pb-6">
                  <CardTitle className="text-xl">{plan.name}</CardTitle>
                  <CardDescription className="text-base pt-2 min-h-[40px]">{plan.description}</CardDescription>
                  <div className="pt-4">
                    <span className="text-4xl font-extrabold text-gray-900">{plan.price}</span>
                    <span className="text-gray-500 font-medium">/lifetime</span>
                  </div>
                  <div className="mt-2 text-sm text-gray-500 font-medium">
                    {plan.users}
                  </div>
                </CardHeader>
                <CardContent className="flex-grow">
                  <ul className="space-y-4 mb-8">
                    {plan.features.map((feature, i) => (
                      <li key={i} className="flex items-start gap-3">
                        <Check className="h-5 w-5 text-cyan-500 flex-shrink-0 mt-0.5" />
                        <span className="text-gray-600 text-sm">{feature}</span>
                      </li>
                    ))}
                  </ul>
                </CardContent>
                <CardFooter className="pt-0">
                  <Link href="/downloads" className="w-full">
                    <Button 
                      className={`w-full h-12 font-semibold ${plan.highlight ? 'bg-cyan-500 hover:bg-cyan-600 text-white' : 'bg-gray-100 hover:bg-gray-200 text-gray-900'}`}
                    >
                      Buy Now
                    </Button>
                  </Link>
                </CardFooter>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Comparison Table */}
      <section className="py-16 bg-gray-50">
        <div className="container mx-auto px-4">
          <div className="max-w-6xl mx-auto">
            <div className="text-center mb-12">
              <h2 className="text-3xl font-bold text-gray-900 mb-4">
                Compare Editions
              </h2>
              <p className="text-gray-600">Detailed breakdown of Silver and Gold capabilities.</p>
            </div>

            <div className="bg-white rounded-xl shadow-lg border border-gray-200 overflow-hidden">
              <div className="overflow-x-auto">
                <table className="w-full text-left border-collapse min-w-[600px]">
                  <thead>
                    <tr className="bg-gray-100 border-b border-gray-200">
                      <th className="p-6 font-semibold text-gray-900 w-1/3">Features</th>
                      <th className="p-6 font-semibold text-gray-900 text-center w-1/3">Silver</th>
                      <th className="p-6 font-semibold text-cyan-600 text-center w-1/3 bg-cyan-50/50">Gold</th>
                    </tr>
                  </thead>
                  <tbody className="divide-y divide-gray-100">
                    {comparisonData.map((row, i) => (
                      <tr key={i} className="hover:bg-gray-50/50 transition-colors">
                        <td className="p-6 text-gray-700 font-medium">{row.feature}</td>
                        <td className="p-6 text-center text-gray-600 text-sm">{row.silver}</td>
                        <td className="p-6 text-center font-medium text-gray-900 text-sm bg-cyan-50/30">{row.gold}</td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            </div>
            
            <div className="mt-8 text-center">
               <Link href="/contact">
                <Button variant="outline" className="px-6 border-cyan-300 text-cyan-700">
                  Talk to a Tally Expert
                </Button>
               </Link>
            </div>
          </div>
        </div>
      </section>

      {/* CTA */}
      <section className="py-20 bg-gradient-to-r from-cyan-500 to-blue-600 text-white">
        <div className="container mx-auto px-4 text-center">
          <h2 className="text-3xl lg:text-4xl font-bold mb-6">
            Upgrade to TallyPrime Today
          </h2>
          <p className="text-lg text-cyan-50 mb-10 max-w-2xl mx-auto">
            Join millions of businesses worldwide. Experience simplicity, speed, and power.
          </p>
          <div className="flex flex-wrap justify-center gap-4">
            <Link href="/downloads">
              <Button size="lg" variant="outline" className="border-white text-white hover:bg-white hover:text-cyan-700 px-8 h-12 text-lg">
                Download Free Trial
              </Button>
            </Link>
            <Link href="/contact">
              <Button size="lg" className="bg-white text-cyan-700 hover:bg-cyan-50 px-8 h-12 text-lg font-semibold">
                Request Demo
              </Button>
            </Link>
          </div>
        </div>
      </section>
    </>
  );
}