import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import Image from 'next/image';
import {
  CheckCircle2,
  Cloud,
  Smartphone,
  Shield,
  Database,
  Zap,
  RefreshCw,
  Globe,
  Lock,
  Server,
  Clock,
  FileText,
  Users,
  Check,
} from 'lucide-react';
import Link from 'next/link';

export default function TallyOnCloudPage() {
  const features = [
    { icon: Cloud, title: 'Cloud-Based', description: 'Access Tally from anywhere, anytime via web browser' },
    { icon: Shield, title: 'Secure', description: 'Bank-grade security with SSL encryption and firewalls' },
    { icon: RefreshCw, title: 'Auto Backup', description: 'Automatic daily backups to ensure data safety' },
    { icon: Globe, title: 'Multi-Location', description: 'Connect multiple branches on a single platform' },
    { icon: Database, title: 'Real-time Sync', description: 'Instant data synchronization between users' },
    { icon: Smartphone, title: 'Mobile Friendly', description: 'Access reports and data on mobile devices' },
  ];

  const benefits = [
    {
      icon: Zap,
      title: 'No VPN Required',
      description: 'Direct access via browser, no complex networking setup',
    },
    {
      icon: Lock,
      title: 'Data Privacy',
      description: 'Isolated environment ensuring your data remains yours',
    },
    {
      icon: Server,
      title: 'Zero Maintenance',
      description: 'We handle server management and Tally updates',
    },
    {
      icon: Clock,
      title: 'High Performance',
      description: 'Optimized servers for lag-free Tally operations',
    },
  ];

  const plans = [
    {
      id: 'silver',
      name: 'Tally Silver on Cloud',
      price: 'NPR 18,000',
      period: 'per year',
      description: 'Designed for single users needing remote access to Tally.',
      highlight: false,
      badge: 'Popular',
      features: [
        'Single User Access',
        'Web-Based Interface',
        'Remote Desktop Access',
        'Daily Data Backups',
        '24/7 Technical Support',
        'SSL Security',
      ],
    },
  ];

  return (
    <>
      {/* Hero Section - Indigo/Blue Theme */}
      <section className="bg-gradient-to-br from-indigo-600 to-blue-700 text-white py-16 pt-45">
        <div className="container mx-auto px-4">
          <div className="grid lg:grid-cols-2 gap-8 items-center">
            {/* Content */}
            <div>
              <div className="inline-flex items-center gap-2 bg-white/20 backdrop-blur-sm px-4 py-2 rounded-full text-sm mb-4">
                <Cloud className="h-4 w-4" />
                <span>The Power of Tally on the Cloud</span>
              </div>
              <h1 className="text-4xl lg:text-5xl font-bold mb-4">
                Tally on Cloud
              </h1>
              <p className="text-xl text-blue-100 mb-6">
                Experience the freedom of cloud computing. Host your Tally Prime on our secure, high-speed servers and access it from any device, anywhere in the world.
              </p>
              <div className="flex flex-wrap gap-4">
                <Link href="/downloads">
                  <Button size="lg" className="bg-white text-indigo-700 hover:bg-indigo-50">
                    Start Free Trial
                  </Button>
                </Link>
                <Link href="/contact">
                  <Button size="lg" variant="outline" className="border-white text-white hover:bg-white hover:text-indigo-700">
                    Contact Sales
                  </Button>
                </Link>
              </div>
            </div>
            
            {/* Image */}
            <div className="flex items-center justify-center">
              <div className="relative w-full max-w-md aspect-square bg-white/10 rounded-2xl border-2 border-white/20 overflow-hidden">
                <Image
                  src="/Tallyoncloud.png"
                  alt="Tally on Cloud Product"
                  fill
                  className="object-cover"
                  priority
                />
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Features */}
      <section className="py-16 bg-white">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Why Host Tally on Cloud?
            </h2>
            <p className="text-lg text-gray-600 max-w-2xl mx-auto">
              Eliminate hardware costs and geographical limitations.
            </p>
          </div>
          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
            {features.map((feature, index) => (
              <Card key={index} className="border-2 hover:border-indigo-300 transition-all hover:shadow-lg">
                <CardContent className="p-6">
                  <div className="w-12 h-12 bg-indigo-100 rounded-xl flex items-center justify-center mb-4">
                    <feature.icon className="h-6 w-6 text-indigo-600" />
                  </div>
                  <h3 className="text-lg font-semibold text-gray-900 mb-2">
                    {feature.title}
                  </h3>
                  <p className="text-gray-600 text-sm leading-relaxed">
                    {feature.description}
                  </p>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Benefits */}
      <section className="py-16 bg-gray-50">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Cloud Hosting Benefits
            </h2>
            <p className="text-lg text-gray-600 max-w-2xl mx-auto">
              Why modern businesses are moving to the cloud
            </p>
          </div>
          <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-6">
            {benefits.map((benefit, index) => (
              <Card key={index} className="border-2">
                <CardContent className="p-6 text-center">
                  <benefit.icon className="h-10 w-10 text-indigo-600 mx-auto mb-3" />
                  <h3 className="font-semibold text-gray-900 mb-2">
                    {benefit.title}
                  </h3>
                  <p className="text-sm text-gray-600 leading-relaxed">
                    {benefit.description}
                  </p>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* How It Works */}
      <section className="py-16 bg-white">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              How It Works
            </h2>
            <p className="text-lg text-gray-600 max-w-2xl mx-auto">
              Get your Tally on the Cloud in 3 simple steps
            </p>
          </div>
          <div className="max-w-4xl mx-auto">
            <div className="space-y-8">
              {[
                { step: '1', title: 'Choose Plan', description: 'Select Silver or Gold based on your user requirements' },
                { step: '2', title: 'Data Migration', description: 'We migrate your existing Tally data to the cloud server' },
                { step: '3', title: 'Go Live', description: 'Receive login credentials and start working instantly' },
              ].map((item, index) => (
                <div key={index} className="flex gap-6 items-start">
                  <div className="flex-shrink-0">
                    <div className="w-16 h-16 bg-gradient-to-br from-indigo-600 to-blue-700 rounded-xl flex items-center justify-center text-white font-bold text-xl">
                      {item.step}
                    </div>
                  </div>
                  <Card className="flex-1 border-2">
                    <CardContent className="p-6">
                      <h3 className="text-lg font-semibold text-gray-900 mb-2">
                        {item.title}
                      </h3>
                      <p className="text-gray-600">{item.description}</p>
                    </CardContent>
                  </Card>
                </div>
              ))}
            </div>
          </div>
        </div>
      </section>

      {/* Pricing Section (Silver & Gold Only) */}
      <section className="py-16 bg-gray-50">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Choose Your Cloud Plan
            </h2>
            <p className="text-lg text-gray-600">
              Flexible hosting plans for single users and teams.
            </p>
          </div>
          
          <div className="max-w-5xl mx-auto grid md:grid-cols-2 gap-8 items-start">
            {plans.map((plan) => (
              <Card 
                key={plan.id} 
                className={`relative flex flex-col h-full border-2 transition-all ${
                  plan.highlight 
                    ? 'border-indigo-600 shadow-2xl scale-105 z-10' 
                    : 'border-gray-200 shadow-sm'
                }`}
              >
                {plan.highlight && (
                  <div className="absolute top-0 right-0 bg-indigo-600 text-white text-xs font-bold px-3 py-1 rounded-bl-lg">
                    {plan.badge}
                  </div>
                )}
                
                <CardHeader className="pb-4">
                  <div className="flex items-center gap-2 mb-2">
                    {plan.id === 'gold' ? <Users className="h-6 w-6 text-indigo-600" /> : <Database className="h-6 w-6 text-gray-600" />}
                    <CardTitle className="text-xl">{plan.name}</CardTitle>
                  </div>
                  <CardDescription className="text-sm min-h-[40px]">{plan.description}</CardDescription>
                  <div className="pt-4">
                    <span className="text-4xl font-extrabold text-gray-900">{plan.price}</span>
                    <span className="text-gray-500 font-medium"> /{plan.period}</span>
                  </div>
                </CardHeader>
                
                <CardContent className="flex-grow">
                  <ul className="space-y-4">
                    {plan.features.map((feature, index) => (
                      <li key={index} className="flex items-start gap-3">
                        <Check className="h-5 w-5 text-green-500 flex-shrink-0 mt-0.5" />
                        <span className="text-gray-700 text-sm">{feature}</span>
                      </li>
                    ))}
                  </ul>
                </CardContent>

                <div className="p-6 pt-0 mt-auto">
                  <Link href="/downloads" className="block">
                    <Button 
                      className={`w-full h-12 font-semibold ${
                        plan.highlight 
                          ? 'bg-indigo-600 hover:bg-indigo-700 text-white' 
                          : 'bg-white border-2 border-gray-200 text-gray-900 hover:bg-gray-50'
                      }`}
                    >
                      Start Free Trial
                    </Button>
                  </Link>
                </div>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Ideal For */}
      <section className="py-16 bg-white">
        <div className="container mx-auto px-4">
          <div className="max-w-3xl mx-auto text-center">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Ideal For
            </h2>
            <p className="text-lg text-gray-600 mb-8">
              Trusted by professionals and businesses across industries
            </p>
          </div>
          <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-4 max-w-5xl mx-auto">
            {['CA / Accountants', 'Multi-branch Retail', 'Distributors', 'Remote Teams'].map((item, index) => (
              <Card key={index} className="border-2 text-center">
                <CardContent className="p-6">
                  <Cloud className="h-10 w-10 text-indigo-600 mx-auto mb-3" />
                  <h3 className="font-semibold text-gray-900">{item}</h3>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* CTA */}
      <section className="py-16 bg-gradient-to-r from-indigo-600 to-blue-700 text-white">
        <div className="container mx-auto px-4 text-center">
          <h2 className="text-3xl lg:text-4xl font-bold mb-4">
            Transform Your Tally Experience
          </h2>
          <p className="text-lg text-indigo-100 mb-8 max-w-2xl mx-auto">
            Switch to Tally on Cloud today and enjoy the freedom of working from anywhere without compromising on speed or security.
          </p>
          <Link href="/downloads">
            <Button size="lg" variant="outline" className="border-white text-white hover:bg-white hover:text-indigo-700 px-8">
              Start Free Trial
            </Button>
          </Link>
        </div>
      </section>
    </>
  );
}