'use client';

import { useState } from 'react';
import Link from 'next/link';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Download, Lock, CheckCircle2, FileDown, AlertCircle, Loader2 } from 'lucide-react';

const products = [
	{
		id: 'Busy-Setup-13.8',
		name: 'Busy 21 Rel 13.8',
		description: 'Latest version of Busy 21 with enhanced features and bug fixes',
		version: '13.8',
		size: '50 MB',
		requiresRegistration: true,
		icon: '/Busy Icon.png',
	},
	{
		id: 'tally-to-busy',
		name: 'Tally to Busy',
		description: 'Migrate your data from Tally to Busy seamlessly',
		version: '1.5',
		size: '15 MB',
		requiresRegistration: false,
		icon: '/Busy Icon.png',
	},
	{
		id: 'busy-agent',
		name: 'BusyAgent',
		description: 'Real-time data synchronization tool for Busy',
		version: '2.3',
		size: '50 MB',
		requiresRegistration: false,
		icon: '/Busy Icon.png',
	},
];

const features = [
	{ icon: <CheckCircle2 className="w-5 h-5" />, text: 'Complete accounting module' },
	{ icon: <CheckCircle2 className="w-5 h-5" />, text: 'Inventory management' },
	{ icon: <CheckCircle2 className="w-5 h-5" />, text: 'VAT reports and compliance' },
	{ icon: <CheckCircle2 className="w-5 h-5" />, text: 'Multi-user support' },
	{ icon: <CheckCircle2 className="w-5 h-5" />, text: 'Customizable reports' },
	{ icon: <CheckCircle2 className="w-5 h-5" />, text: 'Data backup and restore' },
];

const Downloads = () => {
	const [registrationOpen, setRegistrationOpen] = useState(false);
	const [selectedProduct, setSelectedProduct] = useState<any>(null);
	const [formData, setFormData] = useState({
		fullName: '',
		email: '',
		phone: '',
		company: '',
		address: '',
		city: '',
		gstin: '',
		product: '',
	});
	const [errors, setErrors] = useState<Record<string, string>>({});
	const [loading, setLoading] = useState(false);
	const [success, setSuccess] = useState(false);
	const [downloadStarted, setDownloadStarted] = useState(false);

	const validateForm = () => {
		const newErrors: Record<string, string> = {};

		if (!formData.fullName.trim()) {
			newErrors.fullName = 'Full name is required';
		}

		if (!formData.email.trim()) {
			newErrors.email = 'Email is required';
		} else if (!/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(formData.email)) {
			newErrors.email = 'Please enter a valid email';
		}

		if (!formData.phone.trim()) {
			newErrors.phone = 'Phone number is required';
		} else if (!/^[0-9]{10}$/.test(formData.phone.replace(/[\s-]/g, ''))) {
			newErrors.phone = 'Please enter a valid 10-digit phone number';
		}

		if (!formData.product.trim()) {
			newErrors.product = 'Product is required';
		}

		setErrors(newErrors);
		return Object.keys(newErrors).length === 0;
	};

	const handleDownload = (product: any) => {
		if (product.requiresRegistration) {
			setSelectedProduct(product);
			setFormData({ ...formData, product: product.name });
			setRegistrationOpen(true);
		} else {
			// Direct download for non-registered products
			window.open(`/downloads/${product.id}.exe`, '_blank');
		}
	};

	const handleSubmit = async (e: React.FormEvent) => {
		e.preventDefault();

		if (!validateForm()) return;

		setLoading(true);

		// Send registration data to the server
		const res = await fetch('/api/downloads/register', {
			method: 'POST',
			headers: {
				'Content-Type': 'application/json',
			},
			body: JSON.stringify(formData),
		});

		if (res.ok) {
			// Show success message
			setSuccess(true);
			setDownloadStarted(true);

			// Start download after a short delay
			setTimeout(() => {
				window.open(`/downloads/Busy-Setup-13.8.exe`, '_blank');
				setRegistrationOpen(false);
				setSuccess(false);
				setDownloadStarted(false);
				setFormData({
					fullName: '',
					email: '',
					phone: '',
					company: '',
					address: '',
					city: '',
					gstin: '',
					product: '',
				});
				setLoading(false);
			}, 2000);
		} else {
			const errorData = await res.json();
			setLoading(false);
			setErrors((prev) => ({ ...prev, submit: errorData.message || 'Registration failed' }));
		}
	};

	return (
		<>
			{/* Header Section */}
			<section className="pt-32 pb-20 gradient-hero">
				<div className="container-custom">
					<div className="text-center max-w-3xl mx-auto">
						<span className="badge badge-blue mb-4 inline-block">Downloads</span>
						<h1 className="text-4xl md:text-5xl font-bold mb-6">
							Download{' '}
							<span className="text-gradient-primary">Busy Software</span>
						</h1>
						<p className="text-lg text-gray-600 leading-relaxed">
							Get the latest version of Busy 21 and related tools. Register now to download
							and experience the power of complete accounting solution.
						</p>
					</div>
				</div>
			</section>

			{/* Products Section */}
			<section className="py-20 bg-white">
				<div className="container-custom">
					<div className="grid md:grid-cols-3 gap-8">
						{products.map((product) => (
							<Card
								key={product.id}
								className="card-hover shadow-brand border-2 hover:border-[#005a9f] transition-all duration-300"
							>
								<CardHeader>
									<div className="w-16 h-16 mx-auto gradient-primary rounded-xl flex items-center justify-center mb-4">
										<img
											src={product.icon}
											alt={product.name}
											className="w-12 h-12 object-contain"
										/>
									</div>
									<CardTitle className="text-xl text-center">{product.name}</CardTitle>
									<CardDescription className="text-center">
										{product.description}
									</CardDescription>
								</CardHeader>
								<CardContent>
									<div className="flex items-center justify-between text-sm text-gray-600 mb-4">
										<span>Version: {product.version}</span>
										<span>Size: {product.size}</span>
									</div>

									{product.requiresRegistration && (
										<div className="flex items-center gap-2 text-amber-600 text-sm mb-4 p-3 bg-amber-50 rounded-lg">
											<Lock className="w-4 h-4" />
											<span>Registration required</span>
										</div>
									)}

									<Button
										onClick={() => handleDownload(product)}
										className={`w-full ${
											product.requiresRegistration
												? 'gradient-primary text-white'
												: 'bg-[#e05a5a] hover:bg-[#f37474] text-white'
										} rounded-lg py-6 font-medium transition-colors`}
									>
										<Download className="w-5 h-5 mr-2" />
										{product.requiresRegistration ? 'Register & Download' : 'Download Now'}
									</Button>
								</CardContent>
							</Card>
						))}
					</div>
				</div>
			</section>

			{/* Features Section */}
			<section className="py-20 gradient-hero">
				<div className="container-custom">
					<div className="max-w-3xl mx-auto">
						<div className="text-center mb-12">
							<span className="badge badge-blue mb-4">What's Included</span>
							<h2 className="text-3xl md:text-4xl font-bold mb-4">Busy 21 Features</h2>
							<p className="text-lg text-gray-600">
								Everything you need to manage your business finances effectively.
							</p>
						</div>

						<Card className="shadow-brand-xl">
							<CardContent className="p-8">
								<div className="grid sm:grid-cols-2 gap-4">
									{features.map((feature, index) => (
										<div key={index} className="flex items-center gap-3">
											<div className="text-green-500">{feature.icon}</div>
											<span className="text-gray-700">{feature.text}</span>
										</div>
									))}
								</div>
							</CardContent>
						</Card>
					</div>
				</div>
			</section>

			{/* Info Section */}
			<section className="py-20 bg-white">
				<div className="container-custom">
					<div className="max-w-4xl mx-auto">
						<div className="grid md:grid-cols-2 gap-8">
							<Card className="shadow-brand border-l-4 border-[#005a9f]">
								<CardHeader>
									<CardTitle className="flex items-center gap-2">
										<FileDown className="w-6 h-6 text-[#005a9f]" />
										System Requirements
									</CardTitle>
								</CardHeader>
								<CardContent>
									<ul className="space-y-2 text-sm text-gray-600">
										<li>• Windows 10/11 (64-bit)</li>
										<li>• Minimum 4GB RAM</li>
										<li>• 500MB free disk space</li>
										<li>• 1024 x 768 screen resolution</li>
									</ul>
								</CardContent>
							</Card>

							<Card className="shadow-brand border-l-4 border-[#005a9f]">
								<CardHeader>
									<CardTitle className="flex items-center gap-2">
										<CheckCircle2 className="w-6 h-6 text-[#005a9f]" />
										Support Included
									</CardTitle>
								</CardHeader>
								<CardContent>
									<ul className="space-y-2 text-sm text-gray-600">
										<li>• Free installation support</li>
										<li>• Initial training session</li>
										<li>• 30-day free trial</li>
										<li>• Email and phone support</li>
									</ul>
								</CardContent>
							</Card>
						</div>
					</div>
				</div>
			</section>

			{/* Registration Modal */}
			{registrationOpen && (
				<div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50 p-4 animate-fade-in">
					<Card className="w-full max-w-2xl max-h-[90vh] overflow-y-auto">
						<CardHeader>
							<CardTitle>Register to Download</CardTitle>
							<CardDescription>
								Fill in your details to download {selectedProduct?.name}
							</CardDescription>
						</CardHeader>
						<CardContent>
							{!success ? (
								<form onSubmit={handleSubmit} className="space-y-4">
									<div className="grid sm:grid-cols-2 gap-4">
										<div>
											<Label htmlFor="fullName">Full Name *</Label>
											<Input
												id="fullName"
												value={formData.fullName}
												onChange={(e) =>
													setFormData({ ...formData, fullName: e.target.value })
												}
												placeholder="Enter your full name"
												className={errors.fullName ? 'border-red-500' : ''}
											/>
											{errors.fullName && (
												<p className="text-sm text-red-500 mt-1">{errors.fullName}</p>
											)}
										</div>

										<div>
											<Label htmlFor="email">Email *</Label>
											<Input
												id="email"
												type="email"
												value={formData.email}
												onChange={(e) =>
													setFormData({ ...formData, email: e.target.value })
												}
												placeholder="your@email.com"
												className={errors.email ? 'border-red-500' : ''}
											/>
											{errors.email && (
												<p className="text-sm text-red-500 mt-1">{errors.email}</p>
											)}
										</div>
									</div>

									<div className="grid sm:grid-cols-2 gap-4">
										<div>
											<Label htmlFor="phone">Phone *</Label>
											<Input
												id="phone"
												value={formData.phone}
												onChange={(e) =>
													setFormData({ ...formData, phone: e.target.value })
												}
												placeholder="98XXXXXXXX"
												className={errors.phone ? 'border-red-500' : ''}
											/>
											{errors.phone && (
												<p className="text-sm text-red-500 mt-1">{errors.phone}</p>
											)}
										</div>

										<div>
											<Label htmlFor="company">Company Name</Label>
											<Input
												id="company"
												value={formData.company}
												onChange={(e) =>
													setFormData({ ...formData, company: e.target.value })
												}
												placeholder="Your company name"
											/>
										</div>
									</div>

									<div>
										<Label htmlFor="address">Address</Label>
										<Textarea
											id="address"
											value={formData.address}
											onChange={(e) =>
												setFormData({ ...formData, address: e.target.value })
											}
											placeholder="Your address"
											rows={2}
										/>
									</div>

									<div className="grid sm:grid-cols-2 gap-4">
										<div>
											<Label htmlFor="city">City</Label>
											<Input
												id="city"
												value={formData.city}
												onChange={(e) =>
													setFormData({ ...formData, city: e.target.value })
												}
												placeholder="Your city"
											/>
										</div>

										<div>
											<Label htmlFor="gstin">GSTIN</Label>
											<Input
												id="gstin"
												value={formData.gstin}
												onChange={(e) =>
													setFormData({ ...formData, gstin: e.target.value })
												}
												placeholder="Your GSTIN"
											/>
										</div>
									</div>

									<div>
										<Label htmlFor="product">Product *</Label>
										<Input
											id="product"
											value={formData.product}
											onChange={(e) =>
												setFormData({ ...formData, product: e.target.value })
											}
											readOnly
											className="bg-gray-100"
										/>
									</div>

									{errors.submit && (
										<div className="flex items-center gap-2 text-red-500 bg-red-50 p-3 rounded-lg">
											<AlertCircle className="w-5 h-5" />
											<p className="text-sm">{errors.submit}</p>
										</div>
									)}

									<div className="flex gap-4 pt-4">
										<Button
											type="button"
											variant="outline"
											onClick={() => setRegistrationOpen(false)}
											className="flex-1"
										>
											Cancel
										</Button>
										<Button
											type="submit"
											disabled={loading}
											className="flex-1 gradient-primary text-white"
										>
											{loading ? (
												<>
													<Loader2 className="w-4 h-4 mr-2 animate-spin" />
													Registering...
												</>
											) : (
												'Register & Download'
											)}
										</Button>
									</div>
								</form>
							) : (
								<div className="text-center py-8 animate-fade-in">
									<CheckCircle2 className="w-16 h-16 text-green-500 mx-auto mb-4" />
									<h3 className="text-2xl font-bold mb-2">Registration Successful!</h3>
									<p className="text-gray-600 mb-4">
										Thank you for registering. Your download will start automatically.
									</p>
									{downloadStarted && (
										<div className="flex items-center justify-center gap-2 text-blue-600">
											<Loader2 className="w-5 h-5 animate-spin" />
											<span>Starting download...</span>
										</div>
									)}
								</div>
							)}
						</CardContent>
					</Card>
				</div>
			)}
		</>
	);
};

export default Downloads;
