'use client';

import Link from 'next/link';
import { useEffect, useState } from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Calendar, Clock, ArrowRight, Loader2 } from 'lucide-react';

interface BlogPost {
  id: string;
  title: string;
  slug: string;
  excerpt: string;
  category: string;
  createdAt: string;
  readTime?: number;
  featured?: boolean;
}

const staticBlogPosts: BlogPost[] = [
  {
    id: '1',
    title: 'Complete Guide to VAT in Nepal with Busy Software',
    slug: 'complete-guide-to-vat-nepal-busy-software',
    excerpt: 'Learn how Busy accounting software can help your business stay compliant with Nepal VAT regulations. From registration to filing, we cover everything you need to know.',
    category: 'VAT Compliance',
    createdAt: '2024-01-15',
    readTime: 8,
    featured: true,
  },
  {
    id: '2',
    title: 'Top 10 Benefits of Using Busy Accounting Software',
    slug: 'top-10-benefits-busy-accounting-software',
    excerpt: 'Discover why thousands of businesses in Nepal trust Busy for their accounting needs. From inventory management to financial reporting, explore the key advantages.',
    category: 'Accounting',
    createdAt: '2024-01-10',
    readTime: 6,
    featured: true,
  },
  {
    id: '3',
    title: 'Busy 21 vs Tally: Which is Better for Your Business?',
    slug: 'busy-21-vs-tally-comparison',
    excerpt: 'A comprehensive comparison between Busy 21 and Tally accounting software. Make an informed decision for your business with our detailed analysis.',
    category: 'Comparison',
    createdAt: '2024-01-05',
    readTime: 10,
    featured: false,
  },
  {
    id: '4',
    title: 'How to Migrate from Tally to Busy Successfully',
    slug: 'migrate-tally-to-busy-successfully',
    excerpt: 'Step-by-step guide to migrate your data from Tally to Busy without any data loss. Tips and best practices for a smooth transition.',
    category: 'Migration',
    createdAt: '2024-01-02',
    readTime: 7,
    featured: false,
  },
  {
    id: '5',
    title: 'Inventory Management Best Practices with Busy',
    slug: 'inventory-management-best-practices-busy',
    excerpt: 'Learn how to optimize your inventory management using Busy\'s powerful features. Reduce stockouts, improve efficiency, and boost profitability.',
    category: 'Inventory',
    createdAt: '2023-12-28',
    readTime: 6,
    featured: false,
  },
  {
    id: '6',
    title: 'Understanding Financial Reports in Busy 21',
    slug: 'understanding-financial-reports-busy-21',
    excerpt: 'Master financial reporting with Busy 21. Learn to generate and interpret balance sheets, profit & loss statements, cash flow statements, and more.',
    category: 'Reports',
    createdAt: '2023-12-20',
    readTime: 9,
    featured: false,
  },
];

const Blog = () => {
  const [blogPosts, setBlogPosts] = useState<BlogPost[]>(staticBlogPosts);
  const [loading, setLoading] = useState(true);
  const [selectedCategory, setSelectedCategory] = useState('All');

  useEffect(() => {
    fetchBlogPosts();
  }, []);

  const fetchBlogPosts = async () => {
    try {
      const response = await fetch('/api/admin/blog');
      if (response.ok) {
        const dynamicPosts = await response.json();
        // Combine dynamic posts from dashboard with static posts
        // Calculate read time based on content length
        const postsWithReadTime = dynamicPosts.map((post: any) => ({
          ...post,
          readTime: Math.ceil((post.content?.length || post.excerpt?.length || 0) / 200),
          date: post.createdAt,
        }));
        // Merge with static posts, giving priority to dynamic posts
        const mergedPosts = [...postsWithReadTime, ...staticBlogPosts];
        // Remove duplicates by ID
        const uniquePosts = Array.from(new Map(mergedPosts.map(p => [p.id, p])).values());
        setBlogPosts(uniquePosts);
      }
    } catch (error) {
      console.error('Error fetching blog posts:', error);
      // Fall back to static posts
      setBlogPosts(staticBlogPosts);
    } finally {
      setLoading(false);
    }
  };

  const allCategories = ['All', ...new Set(blogPosts.map(p => p.category))];
  
  const filteredPosts = selectedCategory === 'All' 
    ? blogPosts 
    : blogPosts.filter(p => p.category === selectedCategory);
  return (
    <>
      {/* Header Section */}
      <section className="pt-45 pb-20 gradient-hero">
        <div className="container-custom">
          <div className="text-center max-w-3xl mx-auto">
            <Badge className="badge-blue mb-4">Blog</Badge>
            <h1 className="text-4xl md:text-5xl font-bold mb-6">
              Latest{' '}
              <span className="text-gradient-primary">Insights</span>
            </h1>
            <p className="text-lg text-gray-600 leading-relaxed">
              Stay updated with the latest accounting tips, software updates, and business
              insights from our team of experts.
            </p>
            <div className="mt-8">
              <Link href="/admin">
                <Button className="bg-blue-600 hover:bg-blue-700 text-white font-semibold px-8 py-2">
                  Create a New Post
                </Button>
              </Link>
            </div>
          </div>
        </div>
      </section>

      {loading ? (
        <section className="py-20 flex items-center justify-center">
          <div className="text-center">
            <Loader2 className="w-10 h-10 animate-spin mx-auto mb-4" />
            <p>Loading blog posts...</p>
          </div>
        </section>
      ) : (
        <>
          {/* Category Filter */}
          <section className="py-8 bg-white border-b border-gray-200">
            <div className="container-custom">
              <div className="flex flex-wrap gap-2 justify-center">
                {allCategories.map((category) => (
                  <Button
                    key={category}
                    variant={category === selectedCategory ? 'default' : 'outline'}
                    onClick={() => setSelectedCategory(category)}
                    className={category === selectedCategory ? 'gradient-primary text-white' : 'border-[#005a9f] text-[#005a9f] hover:bg-[#005a9f] hover:text-white'}
                  >
                    {category}
                  </Button>
                ))}
              </div>
            </div>
          </section>

          {/* Featured Posts */}
          {filteredPosts.filter(post => post.featured).length > 0 && (
            <section className="py-20 bg-white">
              <div className="container-custom">
                <h2 className="text-3xl font-bold mb-8">Featured Articles</h2>
                <div className="grid md:grid-cols-2 gap-8">
                  {filteredPosts.filter(post => post.featured).map((post) => (
                    <Card key={post.id} className="card-hover shadow-brand border-2 hover:border-[#005a9f] transition-all duration-300 overflow-hidden">
                      <CardHeader className="p-6">
                        <div className="flex items-center gap-2 mb-4">
                          <Badge className="badge-blue">{post.category}</Badge>
                        </div>
                        <CardTitle className="text-2xl mb-3">{post.title}</CardTitle>
                        <CardDescription className="text-base">{post.excerpt}</CardDescription>
                      </CardHeader>
                      <CardContent className="p-6 pt-0">
                        <div className="flex items-center gap-6 text-sm text-gray-500 mb-4">
                          <div className="flex items-center gap-2">
                            <Calendar className="w-4 h-4" />
                            <span>{new Date(post.createdAt).toLocaleDateString('en-US', { month: 'long', day: 'numeric', year: 'numeric' })}</span>
                          </div>
                          <div className="flex items-center gap-2">
                            <Clock className="w-4 h-4" />
                            <span>{post.readTime || 5} min read</span>
                          </div>
                        </div>
                        <Link href={`/blog/${post.slug}`}>
                          <Button variant="outline" className="w-full border-[#005a9f] text-[#005a9f] hover:bg-[#005a9f] hover:text-white transition-colors">
                            Read Article
                            <ArrowRight className="w-4 h-4 ml-2" />
                          </Button>
                        </Link>
                      </CardContent>
                    </Card>
                  ))}
                </div>
              </div>
            </section>
          )}

          {/* All Posts */}
          <section className="py-20 gradient-hero">
            <div className="container-custom">
              <h2 className="text-3xl font-bold mb-8">
                {selectedCategory === 'All' ? 'All Articles' : `${selectedCategory} Articles`}
              </h2>
              {filteredPosts.length === 0 ? (
                <div className="text-center py-20">
                  <p className="text-gray-500 text-lg">No articles found in this category.</p>
                </div>
              ) : (
                <div className="grid md:grid-cols-3 gap-8">
                  {filteredPosts.map((post) => (
                    <Card key={post.id} className="card-hover shadow-brand border-2 hover:border-[#005a9f] transition-all duration-300">
                      <CardHeader>
                        <div className="flex items-center gap-2 mb-4">
                          <Badge className="badge-blue text-xs">{post.category}</Badge>
                          {post.featured && (
                            <Badge className="badge-saffron text-xs">Featured</Badge>
                          )}
                        </div>
                        <CardTitle className="text-lg">{post.title}</CardTitle>
                        <CardDescription>{post.excerpt}</CardDescription>
                      </CardHeader>
                      <CardContent>
                        <div className="flex items-center gap-4 text-sm text-gray-500 mb-4">
                          <div className="flex items-center gap-2">
                            <Calendar className="w-4 h-4" />
                            <span>{new Date(post.createdAt).toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' })}</span>
                          </div>
                          <div className="flex items-center gap-2">
                            <Clock className="w-4 h-4" />
                            <span>{post.readTime || 5}m</span>
                          </div>
                        </div>
                        <Link href={`/blog/${post.slug}`}>
                          <Button variant="outline" className="w-full border-[#005a9f] text-[#005a9f] hover:bg-[#005a9f] hover:text-white transition-colors text-sm">
                            Read More
                            <ArrowRight className="w-4 h-4 ml-2" />
                          </Button>
                        </Link>
                      </CardContent>
                    </Card>
                  ))}
                </div>
              )}
            </div>
          </section>
        </>
      )}
    </>
  );
};

export default Blog;
