'use client';

import { useEffect, useState } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Mail, Phone, Building2, Trash2, Eye, Loader2 } from 'lucide-react';

interface ContactMessage {
  id: string;
  fullName: string;
  email: string;
  phone?: string;
  company?: string;
  subject: string;
  message: string;
  createdAt: string;
}

const AdminMessages = () => {
  const [messages, setMessages] = useState<ContactMessage[]>([]);
  const [loading, setLoading] = useState(true);
  const [selectedMessage, setSelectedMessage] = useState<ContactMessage | null>(null);
  const [deleting, setDeleting] = useState<string | null>(null);

  useEffect(() => {
    fetchMessages();
  }, []);

  const fetchMessages = async () => {
    try {
      const response = await fetch('/api/contact');
      if (response.ok) {
        const data = await response.json();
        setMessages(data);
      }
    } catch (error) {
      console.error('Error fetching messages:', error);
    } finally {
      setLoading(false);
    }
  };

  const deleteMessage = async (id: string) => {
    if (!confirm('Are you sure you want to delete this message?')) return;

    setDeleting(id);
    try {
      const response = await fetch(`/api/contact/${id}`, {
        method: 'DELETE',
      });

      if (response.ok) {
        setMessages(messages.filter((msg) => msg.id !== id));
        setSelectedMessage(null);
      }
    } catch (error) {
      console.error('Error deleting message:', error);
    } finally {
      setDeleting(null);
    }
  };

  const formatDate = (date: string) => {
    return new Date(date).toLocaleString();
  };

  if (loading) {
    return (
      <div className="pt-32 pb-20 min-h-screen flex items-center justify-center">
        <div className="text-center">
          <Loader2 className="w-10 h-10 animate-spin mx-auto mb-4" />
          <p>Loading messages...</p>
        </div>
      </div>
    );
  }

  return (
    <div className="pt-32 pb-20 min-h-screen bg-gray-50">
      <div className="container-custom">
        <div className="mb-8">
          <h1 className="text-4xl font-bold mb-2">Contact Messages</h1>
          <p className="text-gray-600">Total messages: {messages.length}</p>
        </div>

        <div className="grid lg:grid-cols-3 gap-8">
          {/* Messages List */}
          <div className="lg:col-span-2">
            <Card>
              <CardHeader>
                <CardTitle>All Messages</CardTitle>
                <CardDescription>
                  Click on a message to view details
                </CardDescription>
              </CardHeader>
              <CardContent>
                {messages.length === 0 ? (
                  <p className="text-center text-gray-500 py-8">No messages yet</p>
                ) : (
                  <div className="space-y-3">
                    {messages.map((message) => (
                      <div
                        key={message.id}
                        onClick={() => setSelectedMessage(message)}
                        className={`p-4 border rounded-lg cursor-pointer transition-all ${
                          selectedMessage?.id === message.id
                            ? 'border-blue-500 bg-blue-50'
                            : 'border-gray-200 hover:border-gray-300'
                        }`}
                      >
                        <div className="flex items-start justify-between">
                          <div className="flex-1">
                            <h3 className="font-semibold">{message.fullName}</h3>
                            <p className="text-sm text-gray-600 truncate">{message.subject}</p>
                            <p className="text-xs text-gray-500 mt-1">
                              {formatDate(message.createdAt)}
                            </p>
                          </div>
                          <div className="flex items-center gap-2 ml-4">
                            <Eye className="w-4 h-4 text-gray-400" />
                          </div>
                        </div>
                      </div>
                    ))}
                  </div>
                )}
              </CardContent>
            </Card>
          </div>

          {/* Message Details */}
          <div>
            {selectedMessage ? (
              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">Message Details</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <label className="text-xs font-semibold text-gray-500 uppercase">Name</label>
                    <p className="text-base font-medium">{selectedMessage.fullName}</p>
                  </div>

                  <div>
                    <label className="text-xs font-semibold text-gray-500 uppercase">Email</label>
                    <a
                      href={`mailto:${selectedMessage.email}`}
                      className="text-blue-600 hover:underline flex items-center gap-2"
                    >
                      <Mail className="w-4 h-4" />
                      {selectedMessage.email}
                    </a>
                  </div>

                  {selectedMessage.phone && (
                    <div>
                      <label className="text-xs font-semibold text-gray-500 uppercase">Phone</label>
                      <a
                        href={`tel:${selectedMessage.phone}`}
                        className="text-blue-600 hover:underline flex items-center gap-2"
                      >
                        <Phone className="w-4 h-4" />
                        {selectedMessage.phone}
                      </a>
                    </div>
                  )}

                  {selectedMessage.company && (
                    <div>
                      <label className="text-xs font-semibold text-gray-500 uppercase">Company</label>
                      <p className="text-sm flex items-center gap-2">
                        <Building2 className="w-4 h-4" />
                        {selectedMessage.company}
                      </p>
                    </div>
                  )}

                  <div>
                    <label className="text-xs font-semibold text-gray-500 uppercase">Subject</label>
                    <p className="text-sm font-medium">{selectedMessage.subject}</p>
                  </div>

                  <div>
                    <label className="text-xs font-semibold text-gray-500 uppercase">Message</label>
                    <p className="text-sm text-gray-700 whitespace-pre-wrap bg-gray-50 p-3 rounded">
                      {selectedMessage.message}
                    </p>
                  </div>

                  <div>
                    <label className="text-xs font-semibold text-gray-500 uppercase">Received</label>
                    <p className="text-sm text-gray-600">
                      {formatDate(selectedMessage.createdAt)}
                    </p>
                  </div>

                  <Button
                    onClick={() => deleteMessage(selectedMessage.id)}
                    disabled={deleting === selectedMessage.id}
                    variant="destructive"
                    className="w-full"
                  >
                    {deleting === selectedMessage.id ? (
                      <>
                        <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                        Deleting...
                      </>
                    ) : (
                      <>
                        <Trash2 className="w-4 h-4 mr-2" />
                        Delete Message
                      </>
                    )}
                  </Button>
                </CardContent>
              </Card>
            ) : (
              <Card>
                <CardContent className="py-12">
                  <p className="text-center text-gray-500">
                    Select a message to view details
                  </p>
                </CardContent>
              </Card>
            )}
          </div>
        </div>
      </div>
    </div>
  );
};

export default AdminMessages;
