import React, { useLayoutEffect, useEffect, useRef, useReducer, useState, useCallback, useMemo } from 'react';
import { isEqual, debounce, throttle } from 'lodash-es';
import Cookies from 'js-cookie';
import screenfull from 'screenfull';
import { useSyncExternalStore } from 'use-sync-external-store/shim/index.js';
import { fetchEventSource } from '@microsoft/fetch-event-source';

var _window_navigator, _window;
function isFunction(val) {
    return typeof val === 'function';
}
function isString(val) {
    return typeof val === 'string';
}
const isDev = process.env.NODE_ENV === 'development' || process.env.NODE_ENV === 'test';
const isBrowser = typeof window !== 'undefined';
const isNavigator = typeof navigator !== 'undefined';
function noop() {}
const isIOS = isBrowser && ((_window = window) == null ? void 0 : (_window_navigator = _window.navigator) == null ? void 0 : _window_navigator.userAgent) && /iP(?:ad|hone|od)/.test(window.navigator.userAgent);

const useIsomorphicLayoutEffect = isBrowser ? useLayoutEffect : useEffect;

const useLatest = (value)=>{
    const ref = useRef(value);
    useIsomorphicLayoutEffect(()=>{
        ref.current = value;
    }, [
        value
    ]);
    return ref;
};

function on(obj, ...args) {
    if (obj && obj.addEventListener) {
        obj.addEventListener(...args);
    }
}
function off(obj, ...args) {
    if (obj && obj.removeEventListener) {
        obj.removeEventListener(...args);
    }
}
const defaultWindow =  typeof window !== 'undefined' ? window : undefined;
const defaultDocument =  typeof document !== 'undefined' ? document : undefined;

const defaultOptions$1 = {};
function defaultOnError(e) {
    console.error(e);
}

function getTargetElement(target, defaultElement) {
    if (!isBrowser) {
        return undefined;
    }
    if (!target) {
        return defaultElement;
    }
    let targetElement;
    if (isFunction(target)) {
        targetElement = target();
    } else if ('current' in target) {
        targetElement = target.current;
    } else {
        targetElement = target;
    }
    return targetElement;
}

const updateReducer = (num)=>(num + 1) % 1000000;
function useUpdate() {
    const [, update] = useReducer(updateReducer, 0);
    return update;
}

const useCustomCompareEffect = (effect, deps, depsEqual)=>{
    if (process.env.NODE_ENV !== 'production') {
        if (!Array.isArray(deps) || !deps.length) {
            console.warn('`useCustomCompareEffect` should not be used with no dependencies. Use React.useEffect instead.');
        }
        if (typeof depsEqual !== 'function') {
            console.warn('`useCustomCompareEffect` should be used with depsEqual callback for comparing deps list');
        }
    }
    const ref = useRef(undefined);
    const forceUpdate = useUpdate();
    if (!ref.current) {
        ref.current = deps;
    }
    useIsomorphicLayoutEffect(()=>{
        if (!depsEqual(deps, ref.current)) {
            ref.current = deps;
            forceUpdate();
        }
    });
    // eslint-disable-next-line react-hooks/exhaustive-deps
    useEffect(effect, ref.current);
};

const useDeepCompareEffect = (effect, deps)=>{
    if (process.env.NODE_ENV !== 'production') {
        if (!Array.isArray(deps) || !deps.length) {
            console.warn('`useDeepCompareEffect` should not be used with no dependencies. Use React.useEffect instead.');
        }
    }
    useCustomCompareEffect(effect, deps, isEqual);
};

function useEventListener(eventName, handler, element, options = defaultOptions$1) {
    const savedHandler = useLatest(handler);
    useDeepCompareEffect(()=>{
        const targetElement = getTargetElement(element, defaultWindow);
        if (!(targetElement && targetElement.addEventListener)) {
            return;
        }
        const eventListener = (event)=>savedHandler.current(event);
        on(targetElement, eventName, eventListener, options);
        return ()=>{
            if (!(targetElement && targetElement.removeEventListener)) {
                return;
            }
            off(targetElement, eventName, eventListener);
        };
    }, [
        eventName,
        element,
        options
    ]);
}

const useMount = (fn)=>{
    if (isDev) {
        if (!isFunction(fn)) {
            console.error(`useMount: parameter \`fn\` expected to be a function, but got "${typeof fn}".`);
        }
    }
    useEffect(()=>{
        fn == null ? void 0 : fn();
    // eslint-disable-next-line react-hooks/exhaustive-deps
    }, []);
};

const useActiveElement = ()=>{
    const [active, setActive] = useState(null);
    const listener = useCallback(()=>{
        var _window;
        setActive((_window = window) == null ? void 0 : _window.document.activeElement);
    }, []);
    useEventListener('blur', listener, defaultWindow, true);
    useEventListener('focus', listener, defaultWindow, true);
    useMount(()=>{
        var _window;
        setActive((_window = window) == null ? void 0 : _window.document.activeElement);
    });
    return active;
};

function useMountedState() {
    const mountedRef = useRef(false);
    const get = useCallback(()=>mountedRef.current, []);
    useEffect(()=>{
        mountedRef.current = true;
        return ()=>{
            mountedRef.current = false;
        };
    }, []);
    return get;
}

function asyncGeneratorStep$7(gen, resolve, reject, _next, _throw, key, arg) {
    try {
        var info = gen[key](arg);
        var value = info.value;
    } catch (error) {
        reject(error);
        return;
    }
    if (info.done) {
        resolve(value);
    } else {
        Promise.resolve(value).then(_next, _throw);
    }
}
function _async_to_generator$7(fn) {
    return function() {
        var self = this, args = arguments;
        return new Promise(function(resolve, reject) {
            var gen = fn.apply(self, args);
            function _next(value) {
                asyncGeneratorStep$7(gen, resolve, reject, _next, _throw, "next", value);
            }
            function _throw(err) {
                asyncGeneratorStep$7(gen, resolve, reject, _next, _throw, "throw", err);
            }
            _next(undefined);
        });
    };
}
const useAsyncEffect = (effect, cleanup = noop, deps)=>{
    const mounted = useMountedState();
    useEffect(()=>{
        const execute = ()=>_async_to_generator$7(function*() {
                if (!mounted()) {
                    return;
                }
                yield effect();
            })();
        execute();
        return ()=>{
            cleanup();
        };
    // eslint-disable-next-line react-hooks/exhaustive-deps
    }, deps);
};

const listerOptions = {
    passive: true
};
const useClickOutside = (target, handler, enabled = true)=>{
    const savedHandler = useLatest(handler);
    const listener = (event)=>{
        if (!enabled) {
            return;
        }
        const element = getTargetElement(target);
        if (!element) {
            return;
        }
        const elements = event.composedPath();
        if (element === event.target || elements.includes(element)) {
            return;
        }
        savedHandler.current(event);
    };
    useEventListener('mousedown', listener, defaultWindow, listerOptions);
    useEventListener('touchstart', listener, defaultWindow, listerOptions);
};

function getInitialState$5(key, defaultValue) {
    // Prevent a React hydration mismatch when a default value is provided.
    if (defaultValue !== undefined) {
        return defaultValue;
    }
    if (isBrowser) {
        return Cookies.get(key);
    }
    if (process.env.NODE_ENV !== 'production') {
        console.warn('`useCookie` When server side rendering, defaultValue should be defined to prevent a hydration mismatches.');
    }
    return '';
}
const useCookie = (key, options = defaultOptions$1, defaultValue)=>{
    const [cookieValue, setCookieValue] = useState(getInitialState$5(key, defaultValue));
    useEffect(()=>{
        const getStoredValue = ()=>{
            const raw = Cookies.get(key);
            if (raw !== undefined && raw !== null) {
                return raw;
            } else {
                if (defaultValue === undefined) {
                    Cookies.remove(key);
                } else {
                    Cookies.set(key, defaultValue, options);
                }
                return defaultValue;
            }
        };
        setCookieValue(getStoredValue());
    // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [
        defaultValue,
        key,
        JSON.stringify(options)
    ]);
    const updateCookie = useCallback((newValue)=>{
        const value = isFunction(newValue) ? newValue(cookieValue) : newValue;
        if (value === undefined) {
            Cookies.remove(key);
        } else {
            Cookies.set(key, value, options);
        }
        setCookieValue(value);
    }, // eslint-disable-next-line react-hooks/exhaustive-deps
    [
        key,
        cookieValue,
        JSON.stringify(options)
    ]);
    const refreshCookie = useCallback(()=>{
        const cookieValue = Cookies.get(key);
        if (isString(cookieValue)) {
            setCookieValue(cookieValue);
        }
    }, [
        key
    ]);
    return [
        cookieValue,
        updateCookie,
        refreshCookie
    ];
};

/**
 * keep function reference immutable
 */ const useEvent = (fn)=>{
    if (isDev) {
        if (!isFunction(fn)) {
            console.error(`useEvent expected parameter is a function, got ${typeof fn}`);
        }
    }
    const handlerRef = useRef(fn);
    useIsomorphicLayoutEffect(()=>{
        handlerRef.current = fn;
    }, [
        fn
    ]);
    return useCallback((...args)=>{
        const fn = handlerRef.current;
        return fn(...args);
    }, []);
};

const useInterval = (callback, delay, options = defaultOptions$1)=>{
    const { immediate, controls } = options;
    const savedCallback = useLatest(callback);
    const isActive = useRef(false);
    const timer = useRef(null);
    const clean = ()=>{
        timer.current && clearInterval(timer.current);
    };
    const resume = useEvent(()=>{
        isActive.current = true;
        timer.current = setInterval(()=>savedCallback.current(), delay || 0);
    });
    const pause = useEvent(()=>{
        isActive.current = false;
        clean();
    });
    useEffect(()=>{
        if (immediate) {
            savedCallback.current();
        }
        if (controls) {
            return;
        }
        if (delay !== null) {
            resume();
            return ()=>{
                clean();
            };
        }
        return undefined;
    // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [
        delay,
        immediate
    ]);
    return {
        isActive,
        pause,
        resume
    };
};

function padZero(time) {
    return `${time}`.length < 2 ? `0${time}` : `${time}`;
}
function getHMSTime(timeDiff) {
    if (timeDiff <= 0) {
        return [
            '00',
            '00',
            '00'
        ];
    }
    if (timeDiff > 100 * 3600) {
        return [
            '99',
            '59',
            '59'
        ];
    }
    const hour = Math.floor(timeDiff / 3600);
    const minute = Math.floor((timeDiff - hour * 3600) / 60);
    const second = timeDiff - hour * 3600 - minute * 60;
    return [
        padZero(hour),
        padZero(minute),
        padZero(second)
    ];
}
const useCountDown = (time, format = getHMSTime, callback)=>{
    const [remainTime, setRemainTime] = useState(time);
    const [delay, setDelay] = useState(1000);
    useInterval(()=>{
        if (remainTime <= 0) {
            setDelay(null);
            return;
        }
        setRemainTime(remainTime - 1);
    }, delay);
    useEffect(()=>{
        if (time > 0 && remainTime <= 0) {
            callback && callback();
        }
    }, [
        callback,
        remainTime,
        time
    ]);
    const [hour, minute, secoud] = format(remainTime);
    return [
        hour,
        minute,
        secoud
    ];
};

const useCounter = (initialValue = 0, max = null, min = null)=>{
    // avoid exec init code every render
    const initFunc = ()=>{
        let init = typeof initialValue === 'function' ? initialValue() : initialValue;
        typeof init !== 'number' && console.error(`initialValue has to be a number, got ${typeof initialValue}`);
        if (typeof min === 'number') {
            init = Math.max(init, min);
        } else if (min !== null) {
            console.error(`min has to be a number, got ${typeof min}`);
        }
        if (typeof max === 'number') {
            init = Math.min(init, max);
        } else if (max !== null) {
            console.error(`max has to be a number, got ${typeof max}`);
        }
        return init;
    };
    const [value, setValue] = useState(initFunc);
    const set = useEvent((newState)=>{
        setValue((v)=>{
            let nextValue = typeof newState === 'function' ? newState(v) : newState;
            if (typeof min === 'number') {
                nextValue = Math.max(nextValue, min);
            }
            if (typeof max === 'number') {
                nextValue = Math.min(nextValue, max);
            }
            return nextValue;
        });
    });
    const inc = (delta = 1)=>{
        set((value)=>value + delta);
    };
    const dec = (delta = 1)=>{
        set((value)=>value - delta);
    };
    const reset = ()=>{
        set(initFunc);
    };
    return [
        value,
        set,
        inc,
        dec,
        reset
    ];
};

const defaultOptions = {
    observe: false
};

function getInitialState$4(defaultValue) {
    // Prevent a React hydration mismatch when a default value is provided.
    if (defaultValue !== undefined) {
        return defaultValue;
    }
    if (isBrowser) {
        return '';
    }
    if (process.env.NODE_ENV !== 'production') {
        console.warn('`useCssVar` When server side rendering, defaultValue should be defined to prevent a hydration mismatches.');
    }
    return '';
}
const useCssVar = (prop, target, defaultValue, options = defaultOptions)=>{
    const { observe } = options;
    const [variable, setVariable] = useState(getInitialState$4(defaultValue));
    const observerRef = useRef();
    const set = useCallback((v)=>{
        const element = getTargetElement(target);
        if (element == null ? void 0 : element.style) {
            element == null ? void 0 : element.style.setProperty(prop, v);
            setVariable(v);
        }
    }, [
        prop,
        target
    ]);
    const updateCssVar = useCallback(()=>{
        const element = getTargetElement(target);
        if (element) {
            var _window_getComputedStyle_getPropertyValue;
            const value = (_window_getComputedStyle_getPropertyValue = window.getComputedStyle(element).getPropertyValue(prop)) == null ? void 0 : _window_getComputedStyle_getPropertyValue.trim();
            setVariable(value);
        }
    }, [
        target,
        prop
    ]);
    useEffect(()=>{
        var _window_getComputedStyle_getPropertyValue;
        const element = getTargetElement(target);
        if (!element) {
            return;
        }
        const value = (_window_getComputedStyle_getPropertyValue = window.getComputedStyle(element).getPropertyValue(prop)) == null ? void 0 : _window_getComputedStyle_getPropertyValue.trim();
        /** if var don't has value and defaultValue exist */ if (!value && defaultValue) {
            set(defaultValue);
        } else {
            updateCssVar();
        }
        if (!observe) {
            return;
        }
        observerRef.current = new MutationObserver(updateCssVar);
        observerRef.current.observe(element, {
            attributeFilter: [
                'style',
                'class'
            ]
        });
        return ()=>{
            if (observerRef.current) {
                observerRef.current.disconnect();
            }
        };
    }, [
        observe,
        target,
        updateCssVar,
        set,
        defaultValue,
        prop
    ]);
    return [
        variable,
        set
    ];
};

const useCycleList = (list, i = 0)=>{
    const [index, setIndex] = useState(i);
    const set = (i)=>{
        const length = list.length;
        const nextIndex = ((index + i) % length + length) % length;
        setIndex(nextIndex);
    };
    const next = (i = 1)=>{
        set(i);
    };
    const prev = (i = 1)=>{
        set(-i);
    };
    return [
        list[index],
        next,
        prev
    ];
};

function guessSerializerType(rawInit) {
    return rawInit == null || rawInit === undefined ? 'any' : rawInit instanceof Set ? 'set' : rawInit instanceof Map ? 'map' : rawInit instanceof Date ? 'date' : typeof rawInit === 'boolean' ? 'boolean' : typeof rawInit === 'string' ? 'string' : typeof rawInit === 'object' ? 'object' : Array.isArray(rawInit) ? 'object' : !Number.isNaN(rawInit) ? 'number' : 'any';
}

const StorageSerializers = {
    boolean: {
        read: (v)=>v === 'true',
        write: (v)=>String(v)
    },
    object: {
        read: (v)=>JSON.parse(v),
        write: (v)=>JSON.stringify(v)
    },
    number: {
        read: (v)=>Number.parseFloat(v),
        write: (v)=>String(v)
    },
    any: {
        read: (v)=>v,
        write: (v)=>String(v)
    },
    string: {
        read: (v)=>v,
        write: (v)=>String(v)
    },
    map: {
        read: (v)=>new Map(JSON.parse(v)),
        write: (v)=>JSON.stringify(Array.from(v.entries()))
    },
    set: {
        read: (v)=>new Set(JSON.parse(v)),
        write: (v)=>JSON.stringify(Array.from(v))
    },
    date: {
        read: (v)=>new Date(v),
        write: (v)=>v.toISOString()
    }
};
function getInitialState$3(key, defaultValue, storage, serializer, onError) {
    // Prevent a React hydration mismatch when a default value is provided.
    if (defaultValue !== undefined) {
        return defaultValue;
    }
    if (isBrowser) {
        try {
            const raw = storage == null ? void 0 : storage.getItem(key);
            if (raw !== undefined && raw !== null) {
                return serializer == null ? void 0 : serializer.read(raw);
            }
            return null;
        } catch (error) {
            onError == null ? void 0 : onError(error);
        }
    }
    // A default value has not been provided, and you are rendering on the server, warn of a possible hydration mismatch when defaulting to false.
    if (process.env.NODE_ENV !== 'production') {
        console.warn('`createStorage` When server side rendering, defaultValue should be defined to prevent a hydration mismatches.');
    }
    return null;
}
function useStorage(key, defaultValue, getStorage = ()=>isBrowser ? sessionStorage : undefined, options = defaultOptions$1) {
    let storage;
    const { onError = defaultOnError, effectStorageValue, mountStorageValue, listenToStorageChanges = true } = options;
    const storageValueRef = useLatest(mountStorageValue != null ? mountStorageValue : effectStorageValue);
    const onErrorRef = useLatest(onError);
    try {
        storage = getStorage();
    } catch (err) {
        onErrorRef.current(err);
    }
    const type = guessSerializerType(defaultValue);
    var _options_serializer;
    const serializerRef = useLatest((_options_serializer = options.serializer) != null ? _options_serializer : StorageSerializers[type]);
    const [state, setState] = useState(getInitialState$3(key, defaultValue, storage, serializerRef.current, onError));
    useDeepCompareEffect(()=>{
        const serializer = serializerRef.current;
        const storageValue = storageValueRef.current;
        var _ref;
        const data = (_ref = storageValue ? isFunction(storageValue) ? storageValue() : storageValue : defaultValue) != null ? _ref : null;
        const getStoredValue = ()=>{
            try {
                const raw = storage == null ? void 0 : storage.getItem(key);
                if (raw !== undefined && raw !== null) {
                    return serializer.read(raw);
                } else {
                    storage == null ? void 0 : storage.setItem(key, serializer.write(data));
                    return data;
                }
            } catch (e) {
                onErrorRef.current(e);
            }
        };
        setState(getStoredValue());
    }, [
        key,
        storage
    ]);
    const updateState = useEvent((valOrFunc)=>{
        const currentState = isFunction(valOrFunc) ? valOrFunc(state) : valOrFunc;
        setState(currentState);
        if (currentState === null) {
            storage == null ? void 0 : storage.removeItem(key);
        } else {
            try {
                storage == null ? void 0 : storage.setItem(key, serializerRef.current.write(currentState));
            } catch (e) {
                onErrorRef.current(e);
            }
        }
    });
    const listener = useEvent(()=>{
        try {
            const raw = storage == null ? void 0 : storage.getItem(key);
            if (raw !== undefined && raw !== null) {
                updateState(serializerRef.current.read(raw));
            } else {
                updateState(null);
            }
        } catch (e) {
            onErrorRef.current(e);
        }
    });
    useEffect(()=>{
        if (listenToStorageChanges) {
            window.addEventListener('storage', listener);
            return ()=>window.removeEventListener('storage', listener);
        }
        return ()=>{};
    }, [
        listenToStorageChanges,
        listener
    ]);
    return [
        state,
        updateState
    ];
}

const useColorMode = (options)=>{
    const { selector = 'html', attribute = 'class', modes, defaultValue, storageKey = 'reactuses-color-mode', storage = ()=>isBrowser ? localStorage : undefined, initialValueDetector, modeClassNames = {} } = options;
    const initialValueDetectorRef = useLatest(initialValueDetector);
    // Validate modes array
    if (!modes || modes.length === 0) {
        throw new Error('useColorMode: modes array cannot be empty');
    }
    // Get initial value from detector or use first mode as fallback
    const getInitialValue = useCallback(()=>{
        if (initialValueDetectorRef.current) {
            const initialValueDetector = initialValueDetectorRef.current;
            try {
                const detectedValue = initialValueDetector();
                return modes.includes(detectedValue) ? detectedValue : modes[0];
            } catch (e) {
                return modes[0];
            }
        }
        return defaultValue && modes.includes(defaultValue) ? defaultValue : modes[0];
    // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [
        defaultValue,
        JSON.stringify(modes)
    ]);
    const [colorMode, setColorMode] = useStorage(storageKey, defaultValue, storage, {
        mountStorageValue: getInitialValue
    });
    // Apply color mode to DOM element
    useDeepCompareEffect(()=>{
        var _window;
        if (!colorMode) return;
        const element = (_window = window) == null ? void 0 : _window.document.querySelector(selector);
        if (!element) return;
        // Remove all existing mode classes/attributes
        modes.forEach((mode)=>{
            const className = modeClassNames[mode] || mode;
            if (attribute === 'class') {
                element.classList.remove(className);
            } else {
                element.removeAttribute(attribute);
            }
        });
        // Apply current mode class/attribute
        const currentClassName = modeClassNames[colorMode] || colorMode;
        if (attribute === 'class') {
            element.classList.add(currentClassName);
        } else {
            element.setAttribute(attribute, currentClassName);
        }
        return ()=>{
            // Cleanup: remove current mode class/attribute
            if (attribute === 'class') {
                element.classList.remove(currentClassName);
            } else {
                element.removeAttribute(attribute);
            }
        };
    }, [
        colorMode,
        selector,
        attribute,
        modes,
        modeClassNames
    ]);
    const cycle = useEvent(()=>{
        if (!colorMode) return;
        const currentIndex = modes.indexOf(colorMode);
        const nextIndex = (currentIndex + 1) % modes.length;
        setColorMode(modes[nextIndex]);
    });
    return [
        colorMode,
        setColorMode,
        cycle
    ];
};

function getSystemPreference() {
    return window.matchMedia('(prefers-color-scheme: dark)').matches ? 'dark' : 'light';
}
const useDarkMode = (options)=>{
    const { selector = 'html', attribute = 'class', classNameDark = '', classNameLight = '', storageKey = 'reactuses-color-scheme', storage = ()=>isBrowser ? localStorage : undefined, defaultValue = false } = options;
    // Convert boolean-based options to string-based options for useColorMode
    const [colorMode, setColorMode] = useColorMode({
        selector,
        attribute,
        modes: [
            'light',
            'dark'
        ],
        defaultValue: defaultValue ? 'dark' : 'light',
        storageKey,
        storage,
        initialValueDetector: getSystemPreference,
        modeClassNames: {
            dark: classNameDark,
            light: classNameLight
        }
    });
    // Convert string mode back to boolean
    const dark = colorMode === 'dark';
    // Toggle function that switches between dark and light
    const toggle = useCallback(()=>{
        setColorMode(dark ? 'light' : 'dark');
    }, [
        dark,
        setColorMode
    ]);
    // Set function that accepts boolean value
    const setDark = useCallback((value)=>{
        if (typeof value === 'function') {
            const currentDark = colorMode === 'dark';
            const newDark = value(currentDark);
            if (newDark !== null) {
                setColorMode(newDark ? 'dark' : 'light');
            }
        } else if (value !== null) {
            setColorMode(value ? 'dark' : 'light');
        }
    }, [
        colorMode,
        setColorMode
    ]);
    return [
        dark,
        toggle,
        setDark
    ];
};

function useUnmount(fn) {
    if (isDev) {
        if (!isFunction(fn)) {
            console.error(`useUnmount expected parameter is a function, got ${typeof fn}`);
        }
    }
    const fnRef = useLatest(fn);
    useEffect(()=>()=>{
            fnRef.current();
        }, [
        fnRef
    ]);
}

const useDebounceFn = (fn, wait, options)=>{
    if (isDev) {
        if (!isFunction(fn)) {
            console.error(`useDebounceFn expected parameter is a function, got ${typeof fn}`);
        }
    }
    const fnRef = useLatest(fn);
    const debounced = useMemo(()=>debounce((...args)=>{
            return fnRef.current(...args);
        }, wait, options), // eslint-disable-next-line react-hooks/exhaustive-deps
    [
        JSON.stringify(options),
        wait
    ]);
    useUnmount(()=>{
        debounced.cancel();
    });
    return {
        run: debounced,
        cancel: debounced.cancel,
        flush: debounced.flush
    };
};

const useDebounce = (value, wait, options)=>{
    const [debounced, setDebounced] = useState(value);
    const { run } = useDebounceFn(()=>{
        setDebounced(value);
    }, wait, options);
    useEffect(()=>{
        run();
    }, [
        run,
        value
    ]);
    return debounced;
};

function getInitialState$2(defaultValue) {
    // Prevent a React hydration mismatch when a default value is provided.
    if (defaultValue !== undefined) {
        return defaultValue;
    }
    if (isBrowser) {
        return document.visibilityState;
    }
    if (process.env.NODE_ENV !== 'production') {
        console.warn('`useDocumentVisibility` When server side rendering, defaultValue should be defined to prevent a hydration mismatches.');
    }
    return 'visible';
}
function useDocumentVisibility(defaultValue) {
    const [visible, setVisible] = useState(getInitialState$2(defaultValue));
    useEventListener('visibilitychange', ()=>{
        setVisible(document.visibilityState);
    }, ()=>document);
    useEffect(()=>{
        setVisible(document.visibilityState);
    }, []);
    return visible;
}

const useDoubleClick = ({ target, latency = 300, onSingleClick = ()=>{}, onDoubleClick = ()=>{} })=>{
    const handle = useCallback((onSingleClick, onDoubleClick)=>{
        let count = 0;
        return (e)=>{
            // prevent ios double click slide
            if (e.type === 'touchend') {
                e.stopPropagation();
                e.preventDefault();
            }
            count += 1;
            setTimeout(()=>{
                if (count === 1) {
                    onSingleClick(e);
                } else if (count === 2) {
                    onDoubleClick(e);
                }
                count = 0;
            }, latency);
        };
    }, [
        latency
    ]);
    const handleClick = handle(onSingleClick, onDoubleClick);
    const handleTouchEnd = handle(onSingleClick, onDoubleClick);
    useEventListener('click', handleClick, target);
    useEventListener('touchend', handleTouchEnd, target, {
        passive: false
    });
};

function isScrollX(node) {
    if (!node) {
        return false;
    }
    return getComputedStyle(node).overflowX === 'auto' || getComputedStyle(node).overflowX === 'scroll';
}
function isScrollY(node) {
    if (!node) {
        return false;
    }
    return getComputedStyle(node).overflowY === 'auto' || getComputedStyle(node).overflowY === 'scroll';
}
const useDraggable = (target, options = {})=>{
    const { draggingElement, containerElement } = options;
    var _options_handle;
    const draggingHandle = (_options_handle = options.handle) != null ? _options_handle : target;
    var _options_initialValue;
    const [position, setPositon] = useState((_options_initialValue = options.initialValue) != null ? _options_initialValue : {
        x: 0,
        y: 0
    });
    useDeepCompareEffect(()=>{
        var _options_initialValue;
        setPositon((_options_initialValue = options.initialValue) != null ? _options_initialValue : {
            x: 0,
            y: 0
        });
    }, [
        options.initialValue
    ]);
    const [pressedDelta, setPressedDelta] = useState();
    const filterEvent = (e)=>{
        if (options.pointerTypes) {
            return options.pointerTypes.includes(e.pointerType);
        }
        return true;
    };
    const handleEvent = (e)=>{
        if (options.preventDefault) {
            e.preventDefault();
        }
        if (options.stopPropagation) {
            e.stopPropagation();
        }
    };
    const start = (e)=>{
        var _container_getBoundingClientRect;
        const element = getTargetElement(target);
        if (!filterEvent(e) || !element) {
            return;
        }
        if (options.exact && e.target !== element) {
            return;
        }
        const container = getTargetElement(containerElement);
        const containerRect = container == null ? void 0 : (_container_getBoundingClientRect = container.getBoundingClientRect) == null ? void 0 : _container_getBoundingClientRect.call(container);
        const targetRect = element.getBoundingClientRect();
        const pos = {
            x: e.clientX - (container && containerRect ? targetRect.left - (containerRect == null ? void 0 : containerRect.left) + container.scrollLeft : targetRect.left),
            y: e.clientY - (container && containerRect ? targetRect.top - containerRect.top + container.scrollTop : targetRect.top)
        };
        if ((options.onStart == null ? void 0 : options.onStart.call(options, pos, e)) === false) {
            return;
        }
        setPressedDelta(pos);
        handleEvent(e);
    };
    const move = (e)=>{
        const element = getTargetElement(target);
        if (!filterEvent(e) || !element) {
            return;
        }
        if (!pressedDelta) {
            return;
        }
        const container = getTargetElement(containerElement);
        const targetRect = element.getBoundingClientRect();
        let { x, y } = position;
        x = e.clientX - pressedDelta.x;
        y = e.clientY - pressedDelta.y;
        if (container) {
            const containerWidth = isScrollX(container) ? container.scrollWidth : container.clientWidth;
            const containerHeight = isScrollY(container) ? container.scrollHeight : container.clientHeight;
            x = Math.min(Math.max(0, x), containerWidth - targetRect.width);
            y = Math.min(Math.max(0, y), containerHeight - targetRect.height);
        }
        setPositon({
            x,
            y
        });
        options.onMove == null ? void 0 : options.onMove.call(options, position, e);
        handleEvent(e);
    };
    const end = (e)=>{
        if (!filterEvent(e)) {
            return;
        }
        if (!pressedDelta) {
            return;
        }
        setPressedDelta(undefined);
        options.onEnd == null ? void 0 : options.onEnd.call(options, position, e);
        handleEvent(e);
    };
    useEventListener('pointerdown', start, draggingHandle, true);
    useEventListener('pointermove', move, draggingElement, true);
    useEventListener('pointerup', end, draggingElement, true);
    return [
        position.x,
        position.y,
        !!pressedDelta,
        setPositon
    ];
};

const useDropZone = (target, onDrop)=>{
    const [over, setOver] = useState(false);
    const counter = useRef(0);
    useEventListener('dragenter', (event)=>{
        event.preventDefault();
        counter.current += 1;
        setOver(true);
    }, target);
    useEventListener('dragover', (event)=>{
        event.preventDefault();
    }, target);
    useEventListener('dragleave', (event)=>{
        event.preventDefault();
        counter.current -= 1;
        if (counter.current === 0) {
            setOver(false);
        }
    }, target);
    useEventListener('drop', (event)=>{
        var _event_dataTransfer;
        event.preventDefault();
        counter.current = 0;
        setOver(false);
        var _event_dataTransfer_files;
        const files = Array.from((_event_dataTransfer_files = (_event_dataTransfer = event.dataTransfer) == null ? void 0 : _event_dataTransfer.files) != null ? _event_dataTransfer_files : []);
        onDrop == null ? void 0 : onDrop(files.length === 0 ? null : files);
    }, target);
    return over;
};

const useResizeObserver = (target, callback, options = defaultOptions$1)=>{
    const savedCallback = useLatest(callback);
    const observerRef = useRef();
    const stop = useCallback(()=>{
        if (observerRef.current) {
            observerRef.current.disconnect();
        }
    }, []);
    useDeepCompareEffect(()=>{
        const element = getTargetElement(target);
        if (!element) {
            return;
        }
        observerRef.current = new ResizeObserver(savedCallback.current);
        observerRef.current.observe(element, options);
        return stop;
    }, [
        savedCallback,
        stop,
        target,
        options
    ]);
    return stop;
};

const useElementBounding = (target, options = defaultOptions$1)=>{
    const { reset = true, windowResize = true, windowScroll = true, immediate = true } = options;
    const [height, setHeight] = useState(0);
    const [bottom, setBottom] = useState(0);
    const [left, setLeft] = useState(0);
    const [right, setRight] = useState(0);
    const [top, setTop] = useState(0);
    const [width, setWidth] = useState(0);
    const [x, setX] = useState(0);
    const [y, setY] = useState(0);
    const update = useEvent(()=>{
        const element = getTargetElement(target);
        if (!element) {
            if (reset) {
                setHeight(0);
                setBottom(0);
                setLeft(0);
                setRight(0);
                setTop(0);
                setWidth(0);
                setX(0);
                setY(0);
            }
            return;
        }
        const rect = element.getBoundingClientRect();
        setHeight(rect.height);
        setBottom(rect.bottom);
        setLeft(rect.left);
        setRight(rect.right);
        setTop(rect.top);
        setWidth(rect.width);
        setX(rect.x);
        setY(rect.y);
    });
    useResizeObserver(target, update);
    useEffect(()=>{
        if (immediate) {
            update();
        }
    }, [
        immediate,
        update
    ]);
    useEffect(()=>{
        if (windowScroll) {
            window.addEventListener('scroll', update, {
                passive: true
            });
        }
        if (windowResize) {
            window.addEventListener('resize', update, {
                passive: true
            });
        }
        return ()=>{
            if (windowScroll) {
                window.removeEventListener('scroll', update);
            }
            if (windowResize) {
                window.removeEventListener('resize', update);
            }
        };
    }, [
        update,
        windowResize,
        windowScroll
    ]);
    return {
        height,
        bottom,
        left,
        right,
        top,
        width,
        x,
        y,
        update
    };
};

const useElementSize = (target, options = defaultOptions$1)=>{
    const { box = 'content-box' } = options;
    const [width, setWidth] = useState(0);
    const [height, setHeight] = useState(0);
    useResizeObserver(target, ([entry])=>{
        const boxSize = box === 'border-box' ? entry.borderBoxSize : box === 'content-box' ? entry.contentBoxSize : entry.devicePixelContentBoxSize;
        if (boxSize) {
            setWidth(boxSize.reduce((acc, { inlineSize })=>acc + inlineSize, 0));
            setHeight(boxSize.reduce((acc, { blockSize })=>acc + blockSize, 0));
        } else {
            // fallback
            setWidth(entry.contentRect.width);
            setHeight(entry.contentRect.height);
        }
    }, options);
    return [
        width,
        height
    ];
};

const useIntersectionObserver = (target, callback, options = defaultOptions$1)=>{
    const savedCallback = useLatest(callback);
    const observerRef = useRef();
    const stop = useCallback(()=>{
        if (observerRef.current) {
            observerRef.current.disconnect();
        }
    }, []);
    useDeepCompareEffect(()=>{
        const element = getTargetElement(target);
        if (!element) {
            return;
        }
        observerRef.current = new IntersectionObserver(savedCallback.current, options);
        observerRef.current.observe(element);
        return stop;
    }, [
        options
    ]);
    return stop;
};

const useElementVisibility = (target, options = defaultOptions$1)=>{
    const [visible, setVisible] = useState(false);
    const callback = useCallback((entries)=>{
        const rect = entries[0].boundingClientRect;
        setVisible(rect.top <= (window.innerHeight || document.documentElement.clientHeight) && rect.left <= (window.innerWidth || document.documentElement.clientWidth) && rect.bottom >= 0 && rect.right >= 0);
    }, []);
    const stop = useIntersectionObserver(target, callback, options);
    return [
        visible,
        stop
    ];
};

function useEventEmitter() {
    const listeners = useRef([]);
    const _disposed = useRef(false);
    const _event = useRef((listener)=>{
        listeners.current.push(listener);
        const disposable = {
            dispose: ()=>{
                if (!_disposed.current) {
                    for(let i = 0; i < listeners.current.length; i++){
                        if (listeners.current[i] === listener) {
                            listeners.current.splice(i, 1);
                            return;
                        }
                    }
                }
            }
        };
        return disposable;
    });
    const fire = (arg1, arg2)=>{
        const queue = [];
        for(let i = 0; i < listeners.current.length; i++){
            queue.push(listeners.current[i]);
        }
        for(let i = 0; i < queue.length; i++){
            queue[i].call(undefined, arg1, arg2);
        }
    };
    const dispose = ()=>{
        if (listeners.current.length !== 0) {
            listeners.current.length = 0;
        }
        _disposed.current = true;
    };
    return [
        _event.current,
        fire,
        dispose
    ];
}

function useSupported(callback, sync = false) {
    const [supported, setSupported] = useState(false);
    const effect = sync ? useIsomorphicLayoutEffect : useEffect;
    effect(()=>{
        setSupported(Boolean(callback()));
    }, []);
    return supported;
}

function asyncGeneratorStep$6(gen, resolve, reject, _next, _throw, key, arg) {
    try {
        var info = gen[key](arg);
        var value = info.value;
    } catch (error) {
        reject(error);
        return;
    }
    if (info.done) {
        resolve(value);
    } else {
        Promise.resolve(value).then(_next, _throw);
    }
}
function _async_to_generator$6(fn) {
    return function() {
        var self = this, args = arguments;
        return new Promise(function(resolve, reject) {
            var gen = fn.apply(self, args);
            function _next(value) {
                asyncGeneratorStep$6(gen, resolve, reject, _next, _throw, "next", value);
            }
            function _throw(err) {
                asyncGeneratorStep$6(gen, resolve, reject, _next, _throw, "throw", err);
            }
            _next(undefined);
        });
    };
}
const useEyeDropper = ()=>{
    const isSupported = useSupported(()=>typeof window !== 'undefined' && 'EyeDropper' in window, true);
    const open = useCallback((options = {})=>_async_to_generator$6(function*() {
            if (!isSupported) {
                return {
                    sRGBHex: ''
                };
            }
            const eyeDropper = new window.EyeDropper();
            return eyeDropper.open(options);
        })(), [
        isSupported
    ]);
    return [
        isSupported,
        open
    ];
};

function useFavicon(href, baseUrl = '', rel = 'icon') {
    useEffect(()=>{
        const url = `${baseUrl}${href}`;
        const element = document.head.querySelectorAll(`link[rel*="${rel}"]`);
        element.forEach((el)=>el.href = url);
        if (element.length === 0) {
            const link = document.createElement('link');
            link.rel = rel;
            link.href = url;
            document.getElementsByTagName('head')[0].appendChild(link);
        }
    }, [
        baseUrl,
        href,
        rel
    ]);
}

function asyncGeneratorStep$5(gen, resolve, reject, _next, _throw, key, arg) {
    try {
        var info = gen[key](arg);
        var value = info.value;
    } catch (error) {
        reject(error);
        return;
    }
    if (info.done) {
        resolve(value);
    } else {
        Promise.resolve(value).then(_next, _throw);
    }
}
function _async_to_generator$5(fn) {
    return function() {
        var self = this, args = arguments;
        return new Promise(function(resolve, reject) {
            var gen = fn.apply(self, args);
            function _next(value) {
                asyncGeneratorStep$5(gen, resolve, reject, _next, _throw, "next", value);
            }
            function _throw(err) {
                asyncGeneratorStep$5(gen, resolve, reject, _next, _throw, "throw", err);
            }
            _next(undefined);
        });
    };
}
function _extends$4() {
    _extends$4 = Object.assign || function(target) {
        for(var i = 1; i < arguments.length; i++){
            var source = arguments[i];
            for(var key in source){
                if (Object.prototype.hasOwnProperty.call(source, key)) {
                    target[key] = source[key];
                }
            }
        }
        return target;
    };
    return _extends$4.apply(this, arguments);
}
const DEFAULT_OPTIONS = {
    multiple: true,
    accept: '*'
};
const useFileDialog = (options = defaultOptions$1)=>{
    const [files, setFiles] = useState(null);
    const inputRef = useRef();
    const fileOpenPromiseRef = useRef(null);
    const resolveFileOpenPromiseRef = useRef();
    const initFn = useCallback(()=>{
        if (typeof document === 'undefined') {
            return undefined;
        }
        const input = document.createElement('input');
        input.type = 'file';
        input.onchange = (event)=>{
            const result = event.target;
            setFiles(result.files);
            resolveFileOpenPromiseRef.current == null ? void 0 : resolveFileOpenPromiseRef.current.call(resolveFileOpenPromiseRef, result.files);
        };
        return input;
    }, []);
    inputRef.current = initFn();
    const open = (localOptions)=>_async_to_generator$5(function*() {
            if (!inputRef.current) {
                return;
            }
            const _options = _extends$4({}, DEFAULT_OPTIONS, options, localOptions);
            inputRef.current.multiple = _options.multiple;
            inputRef.current.accept = _options.accept;
            // Only set capture attribute if it's explicitly provided
            if (_options.capture !== undefined) {
                inputRef.current.capture = _options.capture;
            }
            fileOpenPromiseRef.current = new Promise((resolve)=>{
                resolveFileOpenPromiseRef.current = resolve;
            });
            inputRef.current.click();
            return fileOpenPromiseRef.current;
        })();
    const reset = ()=>{
        setFiles(null);
        resolveFileOpenPromiseRef.current == null ? void 0 : resolveFileOpenPromiseRef.current.call(resolveFileOpenPromiseRef, null);
        if (inputRef.current) {
            inputRef.current.value = '';
        }
    };
    return [
        files,
        open,
        reset
    ];
};

const useFirstMountState = ()=>{
    const isFirst = useRef(true);
    if (isFirst.current) {
        isFirst.current = false;
        return true;
    }
    return isFirst.current;
};

const useFocus = (target, initialValue = false)=>{
    const [focus, innerSetFocus] = useState(initialValue);
    useEventListener('focus', ()=>innerSetFocus(true), target);
    useEventListener('blur', ()=>innerSetFocus(false), target);
    const setFocus = (value)=>{
        const element = getTargetElement(target);
        if (!element) {
            return;
        }
        if (!value) {
            element.blur();
        } else if (value) {
            element.focus();
        }
    };
    useMount(()=>{
        setFocus(focus);
    });
    return [
        focus,
        setFocus
    ];
};

const useRafFn = (callback, initiallyActive = true)=>{
    const raf = useRef(null);
    const rafActivity = useRef(false);
    const rafCallback = useLatest(callback);
    const step = useCallback((time)=>{
        if (rafActivity.current) {
            rafCallback.current(time);
            raf.current = requestAnimationFrame(step);
        }
    }, [
        rafCallback
    ]);
    const result = useMemo(()=>[
            ()=>{
                // stop
                if (rafActivity.current) {
                    rafActivity.current = false;
                    raf.current && cancelAnimationFrame(raf.current);
                }
            },
            ()=>{
                // start
                if (!rafActivity.current) {
                    rafActivity.current = true;
                    raf.current = requestAnimationFrame(step);
                }
            },
            ()=>rafActivity.current
        ], [
        step
    ]);
    useEffect(()=>{
        if (initiallyActive) {
            result[1]();
        }
        return result[0];
    }, [
        initiallyActive,
        result
    ]);
    return result;
};

function useFps(options = defaultOptions$1) {
    const [fps, setFps] = useState(0);
    var _options_every;
    const every = (_options_every = options.every) != null ? _options_every : 10;
    const last = useRef(performance.now());
    const ticks = useRef(0);
    useRafFn(()=>{
        ticks.current += 1;
        if (ticks.current >= every) {
            const now = performance.now();
            const diff = now - last.current;
            setFps(Math.round(1000 / (diff / ticks.current)));
            last.current = now;
            ticks.current = 0;
        }
    });
    return fps;
}

const useFullscreen = (target, options = defaultOptions$1)=>{
    const { onExit, onEnter } = options;
    const [state, setState] = useState(false);
    const onChange = ()=>{
        if (screenfull.isEnabled) {
            const { isFullscreen } = screenfull;
            if (isFullscreen) {
                onEnter == null ? void 0 : onEnter();
            } else {
                screenfull.off('change', onChange);
                onExit == null ? void 0 : onExit();
            }
            setState(isFullscreen);
        }
    };
    const enterFullscreen = ()=>{
        const el = getTargetElement(target);
        if (!el) {
            return;
        }
        if (screenfull.isEnabled) {
            try {
                screenfull.request(el);
                screenfull.on('change', onChange);
            } catch (error) {
                console.error(error);
            }
        }
    };
    const exitFullscreen = ()=>{
        if (screenfull.isEnabled) {
            screenfull.exit();
        }
    };
    const toggleFullscreen = ()=>{
        if (state) {
            exitFullscreen();
        } else {
            enterFullscreen();
        }
    };
    useUnmount(()=>{
        if (screenfull.isEnabled) {
            screenfull.off('change', onChange);
        }
    });
    return [
        state,
        {
            enterFullscreen: useEvent(enterFullscreen),
            exitFullscreen: useEvent(exitFullscreen),
            toggleFullscreen: useEvent(toggleFullscreen),
            isEnabled: screenfull.isEnabled
        }
    ];
};

const initCoord = {
    accuracy: 0,
    latitude: Number.POSITIVE_INFINITY,
    longitude: Number.POSITIVE_INFINITY,
    altitude: null,
    altitudeAccuracy: null,
    heading: null,
    speed: null
};
const useGeolocation = (options = defaultOptions$1)=>{
    const { enableHighAccuracy = true, maximumAge = 30000, timeout = 27000 } = options;
    const isSupported = useSupported(()=>navigator && 'geolocation' in navigator);
    const [coordinates, setCoordinates] = useState(initCoord);
    const [locatedAt, setLocatedAt] = useState(null);
    const [error, setError] = useState(null);
    const updatePosition = useCallback((position)=>{
        setCoordinates(position.coords);
        setLocatedAt(position.timestamp);
        setError(null);
    }, []);
    const updateError = useCallback((err)=>{
        setCoordinates(initCoord);
        setLocatedAt(null);
        setError(err);
    }, []);
    useEffect(()=>{
        if (!isSupported) {
            return;
        }
        navigator.geolocation.getCurrentPosition(updatePosition, updateError);
        const watchId = navigator.geolocation.watchPosition(updatePosition, updateError, {
            enableHighAccuracy,
            maximumAge,
            timeout
        });
        return ()=>{
            if (watchId) {
                navigator.geolocation.clearWatch(watchId);
            }
        };
    }, [
        enableHighAccuracy,
        isSupported,
        maximumAge,
        timeout,
        updateError,
        updatePosition
    ]);
    return {
        coordinates,
        locatedAt,
        error,
        isSupported
    };
};

const useHover = (target)=>{
    const [hovered, setHovered] = useState(false);
    const onMouseEnter = useCallback(()=>setHovered(true), []);
    const onMouseLeave = useCallback(()=>setHovered(false), []);
    useEventListener('mouseenter', onMouseEnter, target);
    useEventListener('mouseleave', onMouseLeave, target);
    return hovered;
};

const defaultEvents$1 = [
    'mousemove',
    'mousedown',
    'resize',
    'keydown',
    'touchstart',
    'wheel'
];
const oneMinute = 60e3;
const useIdle = (ms = oneMinute, initialState = false, events = defaultEvents$1)=>{
    const [state, setState] = useState(initialState);
    useEffect(()=>{
        let mounted = true;
        let timeout;
        let localState = state;
        const set = (newState)=>{
            if (mounted) {
                localState = newState;
                setState(newState);
            }
        };
        const onEvent = throttle(()=>{
            if (localState) {
                set(false);
            }
            clearTimeout(timeout);
            timeout = setTimeout(()=>set(true), ms);
        }, 50);
        const onVisibility = ()=>{
            if (!document.hidden) {
                onEvent();
            }
        };
        for(let i = 0; i < events.length; i++){
            on(window, events[i], onEvent);
        }
        on(document, 'visibilitychange', onVisibility);
        timeout = setTimeout(()=>set(true), ms);
        return ()=>{
            mounted = false;
            for(let i = 0; i < events.length; i++){
                off(window, events[i], onEvent);
            }
            off(document, 'visibilitychange', onVisibility);
        };
    // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [
        ms,
        events
    ]);
    return state;
};

function useThrottleFn(fn, wait, options) {
    if (isDev) {
        if (!isFunction(fn)) {
            console.error(`useThrottleFn expected parameter is a function, got ${typeof fn}`);
        }
    }
    const fnRef = useLatest(fn);
    const throttled = useMemo(()=>throttle((...args)=>{
            return fnRef.current(...args);
        }, wait, options), // eslint-disable-next-line react-hooks/exhaustive-deps
    [
        wait,
        JSON.stringify(options)
    ]);
    useUnmount(()=>{
        throttled.cancel();
    });
    return {
        run: throttled,
        cancel: throttled.cancel,
        flush: throttled.flush
    };
}

/**
 * We have to check if the scroll amount is close enough to some threshold in order to
 * more accurately calculate arrivedState. This is because scrollTop/scrollLeft are non-rounded
 * numbers, while scrollHeight/scrollWidth and clientHeight/clientWidth are rounded.
 * https://developer.mozilla.org/en-US/docs/Web/API/Element/scrollHeight#determine_if_an_element_has_been_totally_scrolled
 */ const ARRIVED_STATE_THRESHOLD_PIXELS = 1;
const defaultListerOptions = {
    capture: false,
    passive: true
};
const useScroll = (target, options = defaultOptions$1)=>{
    const { throttle = 0, idle = 200, onStop = noop, onScroll = noop, offset = {
        left: 0,
        right: 0,
        top: 0,
        bottom: 0
    }, eventListenerOptions = defaultListerOptions } = options;
    const [x, setX] = useState(0);
    const [y, setY] = useState(0);
    const [isScrolling, setIsScrolling] = useState(false);
    const [arrivedState, setArrivedState] = useState({
        left: true,
        right: false,
        top: true,
        bottom: false
    });
    const [directions, setDirections] = useState({
        left: false,
        right: false,
        top: false,
        bottom: false
    });
    const { run: onScrollEnd } = useDebounceFn((e)=>{
        setIsScrolling(false);
        setDirections({
            left: false,
            right: false,
            top: false,
            bottom: false
        });
        onStop(e);
    }, throttle + idle);
    const onScrollHandler = useEvent((e)=>{
        const eventTarget = e.target === document ? e.target.documentElement : e.target;
        const scrollLeft = eventTarget.scrollLeft;
        let scrollTop = eventTarget.scrollTop;
        // patch for mobile compatible
        if (e.target === document && !scrollTop) scrollTop = document.body.scrollTop;
        setX(scrollLeft);
        setY(scrollTop);
        setDirections({
            left: scrollLeft < x,
            right: scrollLeft > x,
            top: scrollTop < y,
            bottom: scrollTop > y
        });
        setArrivedState({
            left: scrollLeft <= 0 + (offset.left || 0),
            right: scrollLeft + eventTarget.clientWidth >= eventTarget.scrollWidth - (offset.right || 0) - ARRIVED_STATE_THRESHOLD_PIXELS,
            top: scrollTop <= 0 + (offset.top || 0),
            bottom: scrollTop + eventTarget.clientHeight >= eventTarget.scrollHeight - (offset.bottom || 0) - ARRIVED_STATE_THRESHOLD_PIXELS
        });
        setIsScrolling(true);
        onScrollEnd(e);
        onScroll(e);
    });
    const { run: throttleOnScroll } = useThrottleFn(onScrollHandler, throttle);
    useEventListener('scroll', throttle ? throttleOnScroll : onScrollHandler, target, eventListenerOptions);
    return [
        x,
        y,
        isScrolling,
        arrivedState,
        directions
    ];
};

const createUpdateEffect = (hook)=>(effect, deps)=>{
        const isFirstMount = useFirstMountState();
        hook(()=>{
            if (!isFirstMount) {
                return effect();
            }
        }, deps);
    };

const useUpdateEffect = createUpdateEffect(useEffect);

function asyncGeneratorStep$4(gen, resolve, reject, _next, _throw, key, arg) {
    try {
        var info = gen[key](arg);
        var value = info.value;
    } catch (error) {
        reject(error);
        return;
    }
    if (info.done) {
        resolve(value);
    } else {
        Promise.resolve(value).then(_next, _throw);
    }
}
function _async_to_generator$4(fn) {
    return function() {
        var self = this, args = arguments;
        return new Promise(function(resolve, reject) {
            var gen = fn.apply(self, args);
            function _next(value) {
                asyncGeneratorStep$4(gen, resolve, reject, _next, _throw, "next", value);
            }
            function _throw(err) {
                asyncGeneratorStep$4(gen, resolve, reject, _next, _throw, "throw", err);
            }
            _next(undefined);
        });
    };
}
function _extends$3() {
    _extends$3 = Object.assign || function(target) {
        for(var i = 1; i < arguments.length; i++){
            var source = arguments[i];
            for(var key in source){
                if (Object.prototype.hasOwnProperty.call(source, key)) {
                    target[key] = source[key];
                }
            }
        }
        return target;
    };
    return _extends$3.apply(this, arguments);
}
const useInfiniteScroll = (target, onLoadMore, options = defaultOptions$1)=>{
    const savedLoadMore = useLatest(onLoadMore);
    var _options_direction;
    const direction = (_options_direction = options.direction) != null ? _options_direction : 'bottom';
    var _options_distance;
    const state = useScroll(target, _extends$3({}, options, {
        offset: _extends$3({
            [direction]: (_options_distance = options.distance) != null ? _options_distance : 0
        }, options.offset)
    }));
    const di = state[3][direction];
    useUpdateEffect(()=>{
        const element = getTargetElement(target);
        const fn = ()=>_async_to_generator$4(function*() {
                var _element_scrollHeight, _element_scrollWidth;
                const previous = {
                    height: (_element_scrollHeight = element == null ? void 0 : element.scrollHeight) != null ? _element_scrollHeight : 0,
                    width: (_element_scrollWidth = element == null ? void 0 : element.scrollWidth) != null ? _element_scrollWidth : 0
                };
                yield savedLoadMore.current(state);
                if (options.preserveScrollPosition && element) {
                    element.scrollTo({
                        top: element.scrollHeight - previous.height,
                        left: element.scrollWidth - previous.width
                    });
                }
            })();
        fn();
    }, [
        di,
        options.preserveScrollPosition,
        target
    ]);
};

const defaultEvents = [
    'mousedown',
    'mouseup',
    'keydown',
    'keyup'
];
const useKeyModifier = (modifier, options = defaultOptions$1)=>{
    const { events = defaultEvents, initial = false } = options;
    const [state, setState] = useState(initial);
    useMount(()=>{
        events.forEach((listenEvent)=>{
            on(document, listenEvent, (evt)=>{
                if (typeof evt.getModifierState === 'function') {
                    setState(evt.getModifierState(modifier));
                }
            });
        });
        return ()=>{
            events.forEach((listenerEvent)=>{
                off(document, listenerEvent, (evt)=>{
                    if (typeof evt.getModifierState === 'function') {
                        setState(evt.getModifierState(modifier));
                    }
                });
            });
        };
    });
    return state;
};

function useLocalStorage(key, defaultValue, options = defaultOptions$1) {
    return useStorage(key, defaultValue, ()=>isBrowser ? localStorage : undefined, options);
}

function subscribe$1(callback) {
    window.addEventListener('popstate', callback);
    window.addEventListener('hashchange', callback);
    return ()=>{
        window.removeEventListener('popstate', callback);
        window.removeEventListener('hashchange', callback);
    };
}
const useLocationSelector = (selector, /**
   * @description server fallback
   * @default undefined
   */ fallback)=>{
    return useSyncExternalStore(subscribe$1, ()=>selector(location), ()=>fallback);
};

function isTouchEvent(ev) {
    return 'touches' in ev;
}
function preventDefault$1(ev) {
    if (!isTouchEvent(ev)) {
        return;
    }
    if (ev.touches.length < 2 && ev.preventDefault) {
        ev.preventDefault();
    }
}
const useLongPress = (callback, { isPreventDefault = true, delay = 300 } = defaultOptions$1)=>{
    const timeout = useRef();
    const target = useRef();
    const start = useCallback((event)=>{
        // prevent ghost click on mobile devices
        if (isPreventDefault && event.target) {
            on(event.target, 'touchend', preventDefault$1, {
                passive: false
            });
            target.current = event.target;
        }
        timeout.current = setTimeout(()=>callback(event), delay);
    }, [
        callback,
        delay,
        isPreventDefault
    ]);
    const clear = useCallback(()=>{
        // clearTimeout and removeEventListener
        timeout.current && clearTimeout(timeout.current);
        if (isPreventDefault && target.current) {
            off(target.current, 'touchend', preventDefault$1);
        }
    }, [
        isPreventDefault
    ]);
    return {
        onMouseDown: (e)=>start(e),
        onTouchStart: (e)=>start(e),
        onMouseUp: clear,
        onMouseLeave: clear,
        onTouchEnd: clear
    };
};

const defaultState$1 = {
    x: 0,
    y: 0,
    width: 0,
    height: 0,
    top: 0,
    left: 0,
    bottom: 0,
    right: 0
};
const useMeasure = (target, options = defaultOptions$1)=>{
    const [rect, setRect] = useState(defaultState$1);
    const stop = useResizeObserver(target, (entries)=>{
        if (entries[0]) {
            const { x, y, width, height, top, left, bottom, right } = entries[0].contentRect;
            setRect({
                x,
                y,
                width,
                height,
                top,
                left,
                bottom,
                right
            });
        }
    }, options);
    return [
        rect,
        stop
    ];
};

function asyncGeneratorStep$3(gen, resolve, reject, _next, _throw, key, arg) {
    try {
        var info = gen[key](arg);
        var value = info.value;
    } catch (error) {
        reject(error);
        return;
    }
    if (info.done) {
        resolve(value);
    } else {
        Promise.resolve(value).then(_next, _throw);
    }
}
function _async_to_generator$3(fn) {
    return function() {
        var self = this, args = arguments;
        return new Promise(function(resolve, reject) {
            var gen = fn.apply(self, args);
            function _next(value) {
                asyncGeneratorStep$3(gen, resolve, reject, _next, _throw, "next", value);
            }
            function _throw(err) {
                asyncGeneratorStep$3(gen, resolve, reject, _next, _throw, "throw", err);
            }
            _next(undefined);
        });
    };
}
const defaultConstints = {
    audio: true,
    video: true
};
const useMediaDevices = (options = {})=>{
    const { requestPermissions, constraints = defaultConstints } = options;
    const [state, setState] = useState({
        devices: []
    });
    const isSupported = useSupported(()=>navigator && navigator.mediaDevices && navigator.mediaDevices.enumerateDevices);
    const permissionGranted = useRef(false);
    const stream = useRef(null);
    const onChange = useCallback(()=>{
        navigator.mediaDevices.enumerateDevices().then((devices)=>{
            if (stream.current) {
                stream.current.getTracks().forEach((t)=>t.stop());
                stream.current = null;
            }
            setState({
                devices: devices.map(({ deviceId, groupId, kind, label })=>({
                        deviceId,
                        groupId,
                        kind,
                        label
                    }))
            });
        }).catch(noop);
    }, []);
    const ensurePermissions = useCallback(()=>_async_to_generator$3(function*() {
            if (!isSupported) {
                return false;
            }
            if (permissionGranted.current) {
                return true;
            }
            let state;
            try {
                state = (yield navigator.permissions.query({
                    name: 'camera'
                })).state;
            } catch (error) {
                state = 'prompt';
            }
            if (state !== 'granted') {
                stream.current = yield navigator.mediaDevices.getUserMedia(constraints);
                onChange();
                permissionGranted.current = true;
            } else {
                permissionGranted.current = false;
            }
            return permissionGranted.current;
        })(), [
        onChange,
        isSupported,
        constraints
    ]);
    useEffect(()=>{
        if (!isSupported) {
            return;
        }
        if (requestPermissions) {
            ensurePermissions();
        }
        on(navigator.mediaDevices, 'devicechange', onChange);
        onChange();
        return ()=>{
            off(navigator.mediaDevices, 'devicechange', onChange);
        };
    }, [
        onChange,
        isSupported,
        requestPermissions,
        ensurePermissions
    ]);
    return [
        state,
        ensurePermissions
    ];
};

function getInitialState$1(query, defaultState) {
    // Prevent a React hydration mismatch when a default value is provided by not defaulting to window.matchMedia(query).matches.
    if (defaultState !== undefined) {
        return defaultState;
    }
    if (isBrowser) {
        return window.matchMedia(query).matches;
    }
    // A default value has not been provided, and you are rendering on the server, warn of a possible hydration mismatch when defaulting to false.
    if (process.env.NODE_ENV !== 'production') {
        console.warn('`useMediaQuery` When server side rendering, defaultState should be defined to prevent a hydration mismatches.');
    }
    return false;
}
const useMediaQuery = (query, defaultState)=>{
    const [state, setState] = useState(getInitialState$1(query, defaultState));
    useEffect(()=>{
        let mounted = true;
        const mql = window.matchMedia(query);
        const onChange = ()=>{
            if (!mounted) {
                return;
            }
            setState(!!mql.matches);
        };
        if ('addEventListener' in mql) {
            mql.addEventListener('change', onChange);
        } else {
            mql.addListener == null ? void 0 : mql.addListener.call(mql, onChange);
        }
        setState(mql.matches);
        return ()=>{
            mounted = false;
            if ('removeEventListener' in mql) {
                mql.removeEventListener('change', onChange);
            } else {
                mql.removeListener == null ? void 0 : mql.removeListener.call(mql, onChange);
            }
        };
    }, [
        query
    ]);
    return state;
};

function useRafState(initialState) {
    const frame = useRef(0);
    const [state, setState] = useState(initialState);
    const pendingUpdates = useRef([]);
    const setRafState = useCallback((value)=>{
        pendingUpdates.current.push(value);
        cancelAnimationFrame(frame.current);
        frame.current = requestAnimationFrame(()=>{
            const updates = pendingUpdates.current.splice(0);
            setState((prevState)=>{
                let newState = prevState;
                for (const update of updates){
                    newState = typeof update === 'function' ? update(newState) : update;
                }
                return newState;
            });
        });
    }, []);
    useUnmount(()=>{
        cancelAnimationFrame(frame.current);
    });
    return [
        state,
        setRafState
    ];
}

const initState = {
    screenX: Number.NaN,
    screenY: Number.NaN,
    clientX: Number.NaN,
    clientY: Number.NaN,
    pageX: Number.NaN,
    pageY: Number.NaN,
    elementX: Number.NaN,
    elementY: Number.NaN,
    elementH: Number.NaN,
    elementW: Number.NaN,
    elementPosX: Number.NaN,
    elementPosY: Number.NaN
};
const useMouse = (target)=>{
    const [state, setState] = useRafState(initState);
    useEventListener('mousemove', (event)=>{
        const { screenX, screenY, clientX, clientY, pageX, pageY } = event;
        const newState = {
            screenX,
            screenY,
            clientX,
            clientY,
            pageX,
            pageY,
            elementX: Number.NaN,
            elementY: Number.NaN,
            elementH: Number.NaN,
            elementW: Number.NaN,
            elementPosX: Number.NaN,
            elementPosY: Number.NaN
        };
        const targetElement = getTargetElement(target);
        if (targetElement) {
            const { left, top, width, height } = targetElement.getBoundingClientRect();
            newState.elementPosX = left + window.pageXOffset;
            newState.elementPosY = top + window.pageYOffset;
            newState.elementX = pageX - newState.elementPosX;
            newState.elementY = pageY - newState.elementPosY;
            newState.elementW = width;
            newState.elementH = height;
        }
        setState(newState);
    }, defaultDocument);
    return state;
};

const listenerOptions$2 = {
    passive: true
};
const useMousePressed = (target, options = defaultOptions$1)=>{
    const { touch = true, drag = true, initialValue = false } = options;
    const [pressed, setPressed] = useState(initialValue);
    const [sourceType, setSourceType] = useState(null);
    const element = getTargetElement(target);
    const onPressed = useCallback((srcType)=>()=>{
            setPressed(true);
            setSourceType(srcType);
        }, []);
    const onReleased = useCallback(()=>{
        setPressed(false);
        setSourceType(null);
    }, []);
    useEventListener('mousedown', onPressed('mouse'), target, listenerOptions$2);
    useEventListener('mouseleave', onReleased, ()=>window, listenerOptions$2);
    useEventListener('mouseup', onReleased, ()=>window, listenerOptions$2);
    useEffect(()=>{
        if (drag) {
            element == null ? void 0 : element.addEventListener('dragstart', onPressed('mouse'), listenerOptions$2);
            element == null ? void 0 : element.addEventListener('drop', onReleased, listenerOptions$2);
            element == null ? void 0 : element.addEventListener('dragend', onReleased, listenerOptions$2);
        }
        if (touch) {
            element == null ? void 0 : element.addEventListener('touchstart', onPressed('touch'), listenerOptions$2);
            element == null ? void 0 : element.addEventListener('touchend', onReleased, listenerOptions$2);
            element == null ? void 0 : element.addEventListener('touchcancel', onReleased, listenerOptions$2);
        }
        return ()=>{
            if (drag) {
                element == null ? void 0 : element.removeEventListener('dragstart', onPressed('mouse'));
                element == null ? void 0 : element.removeEventListener('drop', onReleased);
                element == null ? void 0 : element.removeEventListener('dragend', onReleased);
            }
            if (touch) {
                element == null ? void 0 : element.removeEventListener('touchstart', onPressed('touch'));
                element == null ? void 0 : element.removeEventListener('touchend', onReleased);
                element == null ? void 0 : element.removeEventListener('touchcancel', onReleased);
            }
        };
    }, [
        drag,
        onPressed,
        onReleased,
        touch,
        element
    ]);
    return [
        pressed,
        sourceType
    ];
};

const useMutationObserver = (callback, target, options = defaultOptions$1)=>{
    const callbackRef = useLatest(callback);
    const observerRef = useRef();
    const stop = useCallback(()=>{
        if (observerRef.current) {
            observerRef.current.disconnect();
        }
    }, []);
    useDeepCompareEffect(()=>{
        const element = getTargetElement(target);
        if (!element) {
            return;
        }
        observerRef.current = new MutationObserver(callbackRef.current);
        observerRef.current.observe(element, options);
        return stop;
    }, [
        options
    ]);
    return stop;
};

const nav = isNavigator ? navigator : undefined;
const conn = nav && (nav.connection || nav.mozConnection || nav.webkitConnection);
function getConnectionState(previousState) {
    const online = nav == null ? void 0 : nav.onLine;
    const previousOnline = previousState == null ? void 0 : previousState.online;
    return {
        online,
        previous: previousOnline,
        since: online !== previousOnline ? new Date() : previousState == null ? void 0 : previousState.since,
        downlink: conn == null ? void 0 : conn.downlink,
        downlinkMax: conn == null ? void 0 : conn.downlinkMax,
        effectiveType: conn == null ? void 0 : conn.effectiveType,
        rtt: conn == null ? void 0 : conn.rtt,
        saveData: conn == null ? void 0 : conn.saveData,
        type: conn == null ? void 0 : conn.type
    };
}
const useNetwork = ()=>{
    const [state, setState] = useState(getConnectionState);
    useEffect(()=>{
        const handleStateChange = ()=>{
            setState(getConnectionState);
        };
        on(window, 'online', handleStateChange, {
            passive: true
        });
        on(window, 'offline', handleStateChange, {
            passive: true
        });
        if (conn) {
            on(conn, 'change', handleStateChange, {
                passive: true
            });
        }
        return ()=>{
            off(window, 'online', handleStateChange);
            off(window, 'offline', handleStateChange);
            if (conn) {
                off(conn, 'change', handleStateChange);
            }
        };
    }, []);
    return state;
};

const useObjectUrl = (object)=>{
    const [url, setUrl] = useState();
    useEffect(()=>{
        if (object) {
            setUrl(URL.createObjectURL(object));
        }
        return ()=>{
            if (url) {
                URL.revokeObjectURL(url);
            }
        };
    // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [
        object
    ]);
    return url;
};

const record = new WeakSet();
const createOnceEffect = (hook)=>(effect, deps)=>{
        const onceWrapper = ()=>{
            const shouldStart = !record.has(effect);
            if (shouldStart) {
                record.add(effect);
                return effect();
            }
        };
        hook(()=>{
            return onceWrapper();
        }, deps);
    };

const useOnceEffect = createOnceEffect(useEffect);

const useOnceLayoutEffect = createOnceEffect(useLayoutEffect);

const useOnline = ()=>{
    const { online } = useNetwork();
    return online;
};

const defaultState = {
    angle: 0,
    type: 'landscape-primary'
};
const useOrientation = (initialState = defaultState)=>{
    const [state, setState] = useState(initialState);
    useEffect(()=>{
        const screen = window.screen;
        let mounted = true;
        const onChange = ()=>{
            if (mounted) {
                const { orientation } = screen;
                if (orientation) {
                    const { angle, type } = orientation;
                    setState({
                        angle,
                        type
                    });
                } else if (window.orientation !== undefined) {
                    setState({
                        angle: typeof window.orientation === 'number' ? window.orientation : 0,
                        type: void 0
                    });
                }
            }
        };
        on(window, 'orientationchange', onChange);
        onChange();
        return ()=>{
            mounted = false;
            off(window, 'orientationchange', onChange);
        };
    }, []);
    const lockOrientation = (type)=>{
        if (isBrowser) {
            return;
        }
        if (!(window && 'screen' in window && 'orientation' in window.screen)) {
            return Promise.reject(new Error('Not supported'));
        }
        return window.screen.orientation.lock(type);
    };
    const unlockOrientation = ()=>{
        if (isBrowser) {
            return;
        }
        if (!(window && 'screen' in window && 'orientation' in window.screen)) {
            return;
        }
        return window.screen.orientation.unlock();
    };
    return [
        state,
        lockOrientation,
        unlockOrientation
    ];
};

function usePageLeave() {
    const [isLeft, setIsLeft] = useState(false);
    const handler = (event)=>{
        if (!window) return;
        event = event || window.event;
        // @ts-expect-error missing types
        const from = event.relatedTarget || event.toElement;
        setIsLeft(!from);
    };
    useEventListener('mouseout', handler, ()=>window, {
        passive: true
    });
    useEventListener('mouseleave', handler, ()=>document, {
        passive: true
    });
    useEventListener('mouseenter', handler, ()=>document, {
        passive: true
    });
    return isLeft;
}

const usePermission = (permissionDesc)=>{
    const [state, setState] = useState('');
    useEffect(()=>{
        var _navigator_permissions;
        const desc = typeof permissionDesc === 'string' ? {
            name: permissionDesc
        } : permissionDesc;
        let mounted = true;
        let permissionStatus = null;
        const onChange = ()=>{
            if (!mounted) {
                return;
            }
            setState(()=>{
                var _permissionStatus_state;
                return (_permissionStatus_state = permissionStatus == null ? void 0 : permissionStatus.state) != null ? _permissionStatus_state : '';
            });
        };
        (_navigator_permissions = navigator.permissions) == null ? void 0 : _navigator_permissions.query(desc).then((status)=>{
            permissionStatus = status;
            on(permissionStatus, 'change', onChange);
            onChange();
        }).catch(noop);
        return ()=>{
            permissionStatus && off(permissionStatus, 'change', onChange);
            mounted = false;
            permissionStatus = null;
        };
    }, [
        permissionDesc
    ]);
    return state;
};

const usePreferredColorScheme = (defaultState = 'no-preference')=>{
    const isLight = useMediaQuery('(prefers-color-scheme: light)', false);
    const isDark = useMediaQuery('(prefers-color-scheme: dark)', false);
    return isDark ? 'dark' : isLight ? 'light' : defaultState;
};

const usePreferredContrast = (defaultState = 'no-preference')=>{
    const isMore = useMediaQuery('(prefexrs-contrast: more)', false);
    const isLess = useMediaQuery('(prefers-contrast: less)', false);
    const isCustom = useMediaQuery('(prefers-contrast: custom)', false);
    return isMore ? 'more' : isLess ? 'less' : isCustom ? 'custom' : defaultState;
};

function usePreferredDark(defaultState) {
    return useMediaQuery('(prefers-color-scheme: dark)', defaultState);
}

// Following these issues I think this is the best way to implement usePrevious:
// https://github.com/childrentime/reactuse/issues/115
// https://github.com/streamich/react-use/issues/2605
// https://github.com/alibaba/hooks/issues/2162
function usePrevious(value) {
    const [current, setCurrent] = useState(value);
    const [previous, setPrevious] = useState();
    if (value !== current) {
        setPrevious(current);
        setCurrent(value);
    }
    return previous;
}

function useReducedMotion(defaultState) {
    return useMediaQuery('(prefers-reduced-motion: reduce)', defaultState);
}

const topVarName = '--reactuse-safe-area-top';
const rightVarName = '--reactuse-safe-area-right';
const bottomVarName = '--reactuse-safe-area-bottom';
const leftVarName = '--reactuse-safe-area-left';
const defaultElement = ()=>document.documentElement;
function useScreenSafeArea() {
    const top = useRef('');
    const right = useRef('');
    const bottom = useRef('');
    const left = useRef('');
    const forceUpdate = useUpdate();
    useCssVar(topVarName, defaultElement, 'env(safe-area-inset-top, 0px)');
    useCssVar(rightVarName, defaultElement, 'env(safe-area-inset-right, 0px)');
    useCssVar(bottomVarName, defaultElement, 'env(safe-area-inset-bottom, 0px)');
    useCssVar(leftVarName, defaultElement, 'env(safe-area-inset-left, 0px)');
    const { run: update } = useDebounceFn(()=>{
        top.current = getValue(topVarName);
        right.current = getValue(rightVarName);
        bottom.current = getValue(bottomVarName);
        left.current = getValue(leftVarName);
        forceUpdate();
    });
    useEffect(()=>{
        update();
    }, [
        update
    ]);
    useEventListener('resize', update);
    return [
        top.current,
        right.current,
        bottom.current,
        left.current,
        update
    ];
}
function getValue(position) {
    return getComputedStyle(document.documentElement).getPropertyValue(position);
}

function _extends$2() {
    _extends$2 = Object.assign || function(target) {
        for(var i = 1; i < arguments.length; i++){
            var source = arguments[i];
            for(var key in source){
                if (Object.prototype.hasOwnProperty.call(source, key)) {
                    target[key] = source[key];
                }
            }
        }
        return target;
    };
    return _extends$2.apply(this, arguments);
}
const initialState = {
    isScratching: false
};
const useScratch = (target, options = {})=>{
    const { disabled = false } = options;
    const [state, setState] = useRafState(initialState);
    const optionsRef = useLatest(options);
    const refState = useRef(state);
    const refScratching = useRef(false);
    const refAnimationFrame = useRef(null);
    const onMoveEvent = (docX, docY)=>{
        if (!refScratching.current) {
            return;
        }
        const el = getTargetElement(target);
        if (!el) {
            return;
        }
        if (refAnimationFrame.current !== null) {
            cancelAnimationFrame(refAnimationFrame.current);
        }
        refAnimationFrame.current = requestAnimationFrame(()=>{
            const { left, top } = el.getBoundingClientRect();
            const elX = left + window.scrollX;
            const elY = top + window.scrollY;
            const x = docX - elX;
            const y = docY - elY;
            setState((oldState)=>{
                const newState = _extends$2({}, oldState, {
                    x,
                    y,
                    dx: x - (oldState.x || 0),
                    dy: y - (oldState.y || 0),
                    end: Date.now(),
                    isScratching: true
                });
                refState.current = newState;
                (optionsRef.current.onScratch || noop)(newState);
                return newState;
            });
        });
    };
    const stopScratching = ()=>{
        if (!refScratching.current) {
            return;
        }
        refScratching.current = false;
        const endState = _extends$2({}, refState.current, {
            isScratching: false
        });
        refState.current = endState;
        (optionsRef.current.onScratchEnd || noop)(endState);
        setState(endState);
    };
    const startScratching = (docX, docY)=>{
        const el = getTargetElement(target);
        if (disabled || !el) {
            return;
        }
        refScratching.current = true;
        const { left, top } = el.getBoundingClientRect();
        const elX = left + window.scrollX;
        const elY = top + window.scrollY;
        const x = docX - elX;
        const y = docY - elY;
        const time = Date.now();
        const newState = {
            isScratching: true,
            start: time,
            end: time,
            docX,
            docY,
            x,
            y,
            dx: 0,
            dy: 0,
            elH: el.offsetHeight,
            elW: el.offsetWidth,
            elX,
            elY,
            posX: elX,
            posY: elY
        };
        refState.current = newState;
        (optionsRef.current.onScratchStart || noop)(newState);
        setState(newState);
    };
    useEventListener('mousedown', (event)=>{
        event.preventDefault();
        startScratching(event.pageX || event.clientX, event.pageY || event.clientY);
    }, target);
    useEventListener('touchstart', (event)=>{
        event.preventDefault();
        startScratching(event.changedTouches[0].pageX || event.changedTouches[0].clientX, event.changedTouches[0].pageY || event.changedTouches[0].clientY);
    }, target);
    useEventListener('mousemove', (event)=>{
        onMoveEvent(event.pageX || event.clientX, event.pageY || event.clientY);
    }, defaultWindow);
    useEventListener('touchmove', (event)=>{
        onMoveEvent(event.changedTouches[0].pageX || event.changedTouches[0].clientX, event.changedTouches[0].pageY || event.changedTouches[0].clientY);
    }, defaultWindow);
    useEventListener('mouseup', stopScratching, defaultWindow);
    useEventListener('touchend', stopScratching, defaultWindow);
    useUnmount(()=>{
        if (refAnimationFrame.current !== null) {
            cancelAnimationFrame(refAnimationFrame.current);
        }
    });
    return state;
};

const useScriptTag = (src, onLoaded = noop, options = defaultOptions$1)=>{
    const { immediate = true, manual = false, type = 'text/javascript', async = true, crossOrigin, referrerPolicy, noModule, defer, attrs = {} } = options;
    const scriptTag = useRef(null);
    const _promise = useRef(null);
    const [status, setStatus] = useState(src ? 'loading' : 'idle');
    /**
   * Load the script specified via `src`.
   *
   * @param waitForScriptLoad Whether if the Promise should resolve once the "load" event is emitted by the <script> attribute, or right after appending it to the DOM.
   * @returns Promise<HTMLScriptElement>
   */ const loadScript = (waitForScriptLoad)=>new Promise((resolve, reject)=>{
            // Some little closure for resolving the Promise.
            const resolveWithElement = (el)=>{
                scriptTag.current = el;
                resolve(el);
                return el;
            };
            // Check if document actually exists, otherwise resolve the Promise (SSR Support).
            if (!document) {
                resolve(false);
                return;
            }
            if (!src) {
                setStatus('idle');
                resolve(false);
                return;
            }
            // Local variable defining if the <script> tag should be appended or not.
            let shouldAppend = false;
            let el = document.querySelector(`script[src="${src}"]`);
            // Script tag not found, preparing the element for appending
            if (!el) {
                el = document.createElement('script');
                el.type = type;
                el.async = async;
                el.src = src;
                // Optional attributes
                if (defer) {
                    el.defer = defer;
                }
                if (crossOrigin) {
                    el.crossOrigin = crossOrigin;
                }
                if (noModule) {
                    el.noModule = noModule;
                }
                if (referrerPolicy) {
                    el.referrerPolicy = referrerPolicy;
                }
                Object.entries(attrs).forEach(([name, value])=>el == null ? void 0 : el.setAttribute(name, value));
                // Enables shouldAppend
                shouldAppend = true;
            } else if (el.hasAttribute('data-loaded')) {
                setStatus(el.getAttribute('data-status'));
                resolveWithElement(el);
            }
            // Event listeners
            el.addEventListener('error', (event)=>{
                setStatus(event.type === 'load' ? 'ready' : 'error');
                const error = new Error(`Failed to load script: ${src}`);
                error.name = 'ScriptLoadError';
                error.event = event;
                return reject(error);
            });
            el.addEventListener('abort', (event)=>{
                setStatus(event.type === 'load' ? 'ready' : 'error');
                const error = new Error(`Script load aborted: ${src}`);
                error.name = 'ScriptLoadAbortError';
                error.event = event;
                return reject(error);
            });
            el.addEventListener('load', (event)=>{
                setStatus(event.type === 'load' ? 'ready' : 'error');
                el.setAttribute('data-loaded', 'true');
                onLoaded(el);
                resolveWithElement(el);
            });
            // Append the <script> tag to head.
            if (shouldAppend) {
                el = document.head.appendChild(el);
            }
            // If script load awaiting isn't needed, we can resolve the Promise.
            if (!waitForScriptLoad) {
                resolveWithElement(el);
            }
        });
    /**
   * Exposed singleton wrapper for `loadScript`, avoiding calling it twice.
   *
   * @param waitForScriptLoad Whether if the Promise should resolve once the "load" event is emitted by the <script> attribute, or right after appending it to the DOM.
   * @returns Promise<HTMLScriptElement>
   */ const load = (waitForScriptLoad = true)=>{
        if (!_promise.current) {
            _promise.current = loadScript(waitForScriptLoad);
        }
        return _promise.current;
    };
    /**
   * Unload the script specified by `src`.
   */ const unload = ()=>{
        if (!document) {
            return;
        }
        _promise.current = null;
        if (scriptTag.current) {
            scriptTag.current = null;
        }
        const el = document.querySelector(`script[src="${src}"]`);
        if (el) {
            document.head.removeChild(el);
        }
    };
    useMount(()=>{
        if (immediate && !manual) {
            load();
        }
    });
    useUnmount(()=>{
        if (!manual) {
            unload();
        }
    });
    return [
        scriptTag.current,
        status,
        load,
        unload
    ];
};

function setScrollParam({ axis, parent, distance }) {
    if (!parent && typeof document === 'undefined') {
        return;
    }
    const method = axis === 'y' ? 'scrollTop' : 'scrollLeft';
    if (parent) {
        parent[method] = distance;
    } else {
        const { body, documentElement } = document;
        body[method] = distance;
        documentElement[method] = distance;
    }
}
function isScrollElement(axis, node) {
    if (!node) {
        return false;
    }
    const AXIS = axis === 'x' ? 'X' : 'Y';
    return getComputedStyle(node)[`overflow${AXIS}`] === 'auto' || getComputedStyle(node)[`overflow${AXIS}`] === 'scroll';
}
const cache = new Map();
function getScrollParent(axis, node) {
    if (!node || !node.parentElement) {
        return null;
    }
    if (cache.has(node)) {
        return cache.get(node) || null;
    }
    let parent = node.parentElement;
    while(parent && !isScrollElement(axis, parent)){
        parent = parent.parentElement;
    }
    if (parent) {
        cache.set(node, parent);
    }
    return parent;
}
function getScrollStart({ axis, parent }) {
    if (!parent && typeof document === 'undefined') {
        return 0;
    }
    const method = axis === 'y' ? 'scrollTop' : 'scrollLeft';
    if (parent) {
        return parent[method];
    }
    const { body, documentElement } = document;
    // while one of it has a value the second is equal 0
    return body[method] + documentElement[method];
}

function easeInOutQuad(t) {
    return t < 0.5 ? 2 * t * t : -1 + (4 - 2 * t) * t;
}

function getRelativePosition({ axis, target, parent, alignment, offset, isList }) {
    if (!target || !parent && typeof document === 'undefined') {
        return 0;
    }
    const isCustomParent = !!parent;
    const parentElement = parent || document.body;
    const parentPosition = parentElement.getBoundingClientRect();
    const targetPosition = target.getBoundingClientRect();
    const getDiff = (property)=>targetPosition[property] - parentPosition[property];
    if (axis === 'y') {
        const diff = getDiff('top');
        if (diff === 0) {
            return 0;
        }
        if (alignment === 'start') {
            const distance = diff - offset;
            const shouldScroll = distance <= targetPosition.height * (isList ? 0 : 1) || !isList;
            return shouldScroll ? distance : 0;
        }
        const parentHeight = isCustomParent ? parentPosition.height : window.innerHeight;
        if (alignment === 'end') {
            const distance = diff + offset - parentHeight + targetPosition.height;
            const shouldScroll = distance >= -targetPosition.height * (isList ? 0 : 1) || !isList;
            return shouldScroll ? distance : 0;
        }
        if (alignment === 'center') {
            return diff - parentHeight / 2 + targetPosition.height / 2;
        }
        return 0;
    }
    if (axis === 'x') {
        const diff = getDiff('left');
        if (diff === 0) {
            return 0;
        }
        if (alignment === 'start') {
            const distance = diff - offset;
            const shouldScroll = distance <= targetPosition.width || !isList;
            return shouldScroll ? distance : 0;
        }
        const parentWidth = isCustomParent ? parentPosition.width : window.innerWidth;
        if (alignment === 'end') {
            const distance = diff + offset - parentWidth + targetPosition.width;
            const shouldScroll = distance >= -targetPosition.width || !isList;
            return shouldScroll ? distance : 0;
        }
        if (alignment === 'center') {
            return diff - parentWidth / 2 + targetPosition.width / 2;
        }
        return 0;
    }
    return 0;
}

const listenerOptions$1 = {
    passive: true
};
const useScrollIntoView = (targetElement, { duration = 1250, axis = 'y', onScrollFinish, easing = easeInOutQuad, offset = 0, cancelable = true, isList = false } = defaultOptions$1, scrollElement)=>{
    const frameID = useRef(0);
    const startTime = useRef(0);
    const shouldStop = useRef(false);
    const reducedMotion = useReducedMotion(false);
    const cancel = ()=>{
        if (frameID.current) {
            cancelAnimationFrame(frameID.current);
        }
    };
    const element = getTargetElement(targetElement);
    const scrollIntoView = ({ alignment = 'start' } = {})=>{
        const parent = getTargetElement(scrollElement) || getScrollParent(axis, element);
        shouldStop.current = false;
        if (frameID.current) {
            cancel();
        }
        var _getScrollStart;
        const start = (_getScrollStart = getScrollStart({
            parent,
            axis
        })) != null ? _getScrollStart : 0;
        const change = getRelativePosition({
            parent,
            target: element,
            axis,
            alignment,
            offset,
            isList
        }) - (parent ? 0 : start);
        const animateScroll = ()=>{
            if (startTime.current === 0) {
                startTime.current = performance.now();
            }
            const now = performance.now();
            const elapsed = now - startTime.current;
            // easing timing progress
            const t = reducedMotion || duration === 0 ? 1 : elapsed / duration;
            const distance = start + change * easing(t);
            setScrollParam({
                parent,
                axis,
                distance
            });
            if (!shouldStop.current && t < 1) {
                frameID.current = requestAnimationFrame(animateScroll);
            } else {
                typeof onScrollFinish === 'function' && onScrollFinish();
                startTime.current = 0;
                frameID.current = 0;
                cancel();
            }
        };
        animateScroll();
    };
    const handleStop = ()=>{
        if (cancelable) {
            shouldStop.current = true;
        }
    };
    useEventListener('wheel', handleStop, null, listenerOptions$1);
    useEventListener('touchmove', handleStop, null, listenerOptions$1);
    useEffect(()=>cancel, []);
    return {
        scrollIntoView,
        cancel
    };
};

function checkOverflowScroll(ele) {
    const style = window.getComputedStyle(ele);
    if (style.overflowX === 'scroll' || style.overflowY === 'scroll' || style.overflowX === 'auto' && ele.clientWidth < ele.scrollWidth || style.overflowY === 'auto' && ele.clientHeight < ele.scrollHeight) {
        return true;
    } else {
        const parent = ele.parentNode;
        if (!parent || parent.tagName === 'BODY') return false;
        return checkOverflowScroll(parent);
    }
}
function preventDefault(rawEvent) {
    const e = rawEvent || window.event;
    const _target = e.target;
    // Do not prevent if element or parentNodes have overflow: scroll set.
    if (checkOverflowScroll(_target)) return false;
    // Do not prevent if the event has more than one touch (usually meaning this is a multi touch gesture like pinch to zoom).
    if (e.touches.length > 1) return true;
    if (e.preventDefault) e.preventDefault();
    return false;
}
const useScrollLock = (target, initialState = false)=>{
    const [locked, setLocked] = useState(initialState);
    const initialOverflowRef = useRef('scroll');
    useEffect(()=>{
        const element = getTargetElement(target);
        if (element) {
            initialOverflowRef.current = element.style.overflow;
            if (locked) {
                element.style.overflow = 'hidden';
            }
        }
    }, [
        locked,
        target
    ]);
    const lock = useEvent(()=>{
        const element = getTargetElement(target);
        if (!element || locked) {
            return;
        }
        if (isIOS) {
            element.addEventListener('touchmove', preventDefault, {
                passive: false
            });
        }
        setLocked(true);
    });
    const unlock = useEvent(()=>{
        const element = getTargetElement(target);
        if (!element || !locked) {
            return;
        }
        if (isIOS) {
            element.removeEventListener('touchmove', preventDefault);
        }
        element.style.overflow = initialOverflowRef.current;
        setLocked(false);
    });
    const set = useEvent((flag)=>{
        if (flag) {
            lock();
        } else {
            unlock();
        }
    });
    return [
        locked,
        set
    ];
};

function useSessionStorage(key, defaultValue, options = defaultOptions$1) {
    return useStorage(key, defaultValue, ()=>isBrowser ? sessionStorage : undefined, options);
}

function _extends$1() {
    _extends$1 = Object.assign || function(target) {
        for(var i = 1; i < arguments.length; i++){
            var source = arguments[i];
            for(var key in source){
                if (Object.prototype.hasOwnProperty.call(source, key)) {
                    target[key] = source[key];
                }
            }
        }
        return target;
    };
    return _extends$1.apply(this, arguments);
}
const useSetState = (initialState)=>{
    const [state, _setState] = useState(initialState);
    const setState = useCallback((statePartial)=>_setState((current)=>_extends$1({}, current, typeof statePartial === 'function' ? statePartial(current) : statePartial)), []);
    return [
        state,
        setState
    ];
};

function useSticky(targetElement, { axis = 'y', nav = 0 }, scrollElement) {
    const [isSticky, setSticky] = useState(false);
    const { run: scrollHandler } = useThrottleFn(()=>{
        const element = getTargetElement(targetElement);
        if (!element) {
            return;
        }
        const rect = element.getBoundingClientRect();
        if (axis === 'y') {
            setSticky((rect == null ? void 0 : rect.top) <= nav);
        } else {
            setSticky((rect == null ? void 0 : rect.left) <= nav);
        }
    }, 50);
    useEffect(()=>{
        const element = getTargetElement(targetElement);
        const scrollParent = getTargetElement(scrollElement) || getScrollParent(axis, element);
        if (!element || !scrollParent) {
            return;
        }
        scrollParent.addEventListener('scroll', scrollHandler);
        scrollHandler();
        return ()=>{
            scrollParent.removeEventListener('scroll', scrollHandler);
        };
    }, [
        axis,
        targetElement,
        scrollElement,
        scrollHandler
    ]);
    return [
        isSticky,
        setSticky
    ];
}

const useTextDirection = (options = defaultOptions$1)=>{
    const { selector = 'html', initialValue = 'ltr' } = options;
    const getValue = ()=>{
        if (initialValue !== undefined) {
            return initialValue;
        }
        if (isBrowser) {
            var _document_querySelector, _document;
            var _document_querySelector_getAttribute;
            return (_document_querySelector_getAttribute = (_document = document) == null ? void 0 : (_document_querySelector = _document.querySelector(selector)) == null ? void 0 : _document_querySelector.getAttribute('dir')) != null ? _document_querySelector_getAttribute : initialValue;
        }
        // A default value has not been provided, and you are rendering on the server, warn of a possible hydration mismatch when defaulting to false.
        if (process.env.NODE_ENV !== 'production') {
            console.warn('`useTextDirection` When server side rendering, defaultState should be defined to prevent a hydration mismatches.');
        }
        return initialValue;
    };
    const [value, setValue] = useState(getValue());
    useEffect(()=>{
        var _document_querySelector, _document;
        var _document_querySelector_getAttribute;
        setValue((_document_querySelector_getAttribute = (_document = document) == null ? void 0 : (_document_querySelector = _document.querySelector(selector)) == null ? void 0 : _document_querySelector.getAttribute('dir')) != null ? _document_querySelector_getAttribute : initialValue);
    }, [
        initialValue,
        selector
    ]);
    const set = (value)=>{
        if (!isBrowser) {
            return;
        }
        if (value !== null) {
            var _document_querySelector;
            (_document_querySelector = document.querySelector(selector)) == null ? void 0 : _document_querySelector.setAttribute('dir', value);
        } else {
            var _document_querySelector1;
            (_document_querySelector1 = document.querySelector(selector)) == null ? void 0 : _document_querySelector1.removeAttribute('dir');
        }
        setValue(value);
    };
    return [
        value,
        set
    ];
};

const useTextSelection = ()=>{
    const [selection, setSelection] = useState(null);
    const forceUpdate = useUpdate();
    const handleSelectionChange = ()=>{
        setSelection(document.getSelection());
        // this is because `document.getSelection` will always return the same object
        forceUpdate();
    };
    useEventListener('selectionchange', handleSelectionChange, ()=>document);
    useEffect(()=>{
        setSelection(document.getSelection());
    }, []);
    return selection;
};

const useThrottle = (value, wait, options)=>{
    const [throttled, setThrottled] = useState(value);
    const { run } = useThrottleFn(()=>{
        setThrottled(value);
    }, wait, options);
    useEffect(()=>{
        run();
    }, [
        run,
        value
    ]);
    return throttled;
};

/**
 * Wrapper for `setTimeout` with controls.
 *
 * @param cb
 * @param interval
 * @param options
 */ const useTimeoutFn = (cb, interval, options = defaultOptions$1)=>{
    const { immediate = true } = options;
    const [pending, setPending] = useState(false);
    const savedCallback = useLatest(cb);
    const timer = useRef();
    const stop = useEvent(()=>{
        // will still be true when component unmount
        setPending(false);
        if (timer.current) {
            clearTimeout(timer.current);
        }
    });
    const start = useEvent((...args)=>{
        if (timer) {
            clearTimeout(timer.current);
        }
        timer.current = setTimeout(()=>{
            setPending(false);
            savedCallback.current(...args);
        }, interval);
        setPending(true);
    });
    useEffect(()=>{
        if (immediate) {
            start();
        }
        return stop;
    }, [
        stop,
        immediate,
        interval,
        start
    ]);
    return [
        pending,
        start,
        stop
    ];
};

const useTimeout = (ms = 0, options = {})=>{
    const update = useUpdate();
    return useTimeoutFn(update, ms, options);
};

const useTitle = (title)=>{
    useEffect(()=>{
        document.title = title;
    }, [
        title
    ]);
};

function toggleReducer(state, nextValue) {
    return typeof nextValue === 'boolean' ? nextValue : !state;
}
const useToggle = (initialValue)=>{
    return useReducer(toggleReducer, initialValue);
};

const useUpdateLayoutEffect = createUpdateEffect(useLayoutEffect);

function asyncGeneratorStep$2(gen, resolve, reject, _next, _throw, key, arg) {
    try {
        var info = gen[key](arg);
        var value = info.value;
    } catch (error) {
        reject(error);
        return;
    }
    if (info.done) {
        resolve(value);
    } else {
        Promise.resolve(value).then(_next, _throw);
    }
}
function _async_to_generator$2(fn) {
    return function() {
        var self = this, args = arguments;
        return new Promise(function(resolve, reject) {
            var gen = fn.apply(self, args);
            function _next(value) {
                asyncGeneratorStep$2(gen, resolve, reject, _next, _throw, "next", value);
            }
            function _throw(err) {
                asyncGeneratorStep$2(gen, resolve, reject, _next, _throw, "throw", err);
            }
            _next(undefined);
        });
    };
}
const useWebNotification = (requestPermissions = false)=>{
    const isSupported = useSupported(()=>!!window && 'Notification' in window);
    const permissionGranted = useRef(false);
    const notificationRef = useRef(null);
    const show = (title, options = defaultOptions$1)=>{
        // If either the browser does not support notifications or the user has
        // not granted permission, do nothing:
        if (!isSupported && !permissionGranted.current) {
            return;
        }
        notificationRef.current = new Notification(title || '', options);
        return notificationRef.current;
    };
    const close = useCallback(()=>{
        if (notificationRef.current) {
            notificationRef.current.close();
        }
        notificationRef.current = null;
    }, []);
    useEffect(()=>{
        permissionGranted.current = isSupported && 'permission' in Notification && Notification.permission === 'granted';
    }, [
        isSupported
    ]);
    const ensurePermissions = useCallback(()=>_async_to_generator$2(function*() {
            if (!isSupported) return;
            if (!permissionGranted.current && Notification.permission !== 'denied') {
                const result = yield Notification.requestPermission();
                if (result === 'granted') permissionGranted.current = true;
            }
            return permissionGranted.current;
        })(), [
        isSupported
    ]);
    useEffect(()=>{
        if (requestPermissions) {
            ensurePermissions();
        }
    }, [
        requestPermissions,
        ensurePermissions
    ]);
    useUnmount(close);
    return {
        isSupported,
        show,
        close,
        ensurePermissions,
        permissionGranted
    };
};

function useWindowsFocus(defauleValue = false) {
    const [focused, setFocused] = useState(defauleValue);
    useEffect(()=>{
        setFocused(window.document.hasFocus());
    }, []);
    useEventListener('blur', ()=>{
        setFocused(false);
    });
    useEventListener('focus', ()=>{
        setFocused(true);
    });
    return focused;
}

const listenerOptions = {
    capture: false,
    passive: true
};
function useWindowScroll() {
    const [state, setState] = useRafState(()=>({
            x: 0,
            y: 0
        }));
    const handleScroll = ()=>{
        setState({
            x: window.scrollX,
            y: window.scrollY
        });
    };
    useEventListener('scroll', handleScroll, defaultWindow, listenerOptions);
    // Set scroll at the first client-side load
    useIsomorphicLayoutEffect(()=>{
        handleScroll();
    }, []);
    return state;
}

function subscribe(callback) {
    window.addEventListener('resize', callback);
    return ()=>{
        window.removeEventListener('resize', callback);
    };
}
const useWindowSize = ()=>{
    const stateDependencies = useRef({}).current;
    const previous = useRef({
        width: 0,
        height: 0
    });
    const isEqual = (prev, current)=>{
        for(const _ in stateDependencies){
            const t = _;
            if (current[t] !== prev[t]) {
                return false;
            }
        }
        return true;
    };
    const cached = useSyncExternalStore(subscribe, ()=>{
        const data = {
            width: window.innerWidth,
            height: window.innerHeight
        };
        if (!isEqual(previous.current, data)) {
            previous.current = data;
            return data;
        }
        return previous.current;
    }, ()=>{
        return previous.current;
    });
    return {
        get width () {
            stateDependencies.width = true;
            return cached.width;
        },
        get height () {
            stateDependencies.height = true;
            return cached.height;
        }
    };
};

function asyncGeneratorStep$1(gen, resolve, reject, _next, _throw, key, arg) {
    try {
        var info = gen[key](arg);
        var value = info.value;
    } catch (error) {
        reject(error);
        return;
    }
    if (info.done) {
        resolve(value);
    } else {
        Promise.resolve(value).then(_next, _throw);
    }
}
function _async_to_generator$1(fn) {
    return function() {
        var self = this, args = arguments;
        return new Promise(function(resolve, reject) {
            var gen = fn.apply(self, args);
            function _next(value) {
                asyncGeneratorStep$1(gen, resolve, reject, _next, _throw, "next", value);
            }
            function _throw(err) {
                asyncGeneratorStep$1(gen, resolve, reject, _next, _throw, "throw", err);
            }
            _next(undefined);
        });
    };
}
const useClipboard = ()=>{
    const [text, setText] = useState('');
    const updateText = useCallback(()=>_async_to_generator$1(function*() {
            // Check if document is focused before attempting to read clipboard
            if (!document.hasFocus()) {
                return;
            }
            try {
                const value = yield window.navigator.clipboard.readText();
                setText(value);
            } catch (error) {
                // Handle cases where clipboard access is denied or unavailable
                console.warn('Failed to read clipboard:', error);
            }
        })(), []);
    useEventListener('copy', updateText);
    useEventListener('cut', updateText);
    // Also listen for focus events to update clipboard when window regains focus
    useEventListener('focus', updateText, window);
    const copy = useCallback((txt)=>_async_to_generator$1(function*() {
            setText(txt);
            try {
                yield window.navigator.clipboard.writeText(txt);
            } catch (error) {
                console.warn('Failed to write to clipboard:', error);
                throw error // Re-throw so caller can handle it
                ;
            }
        })(), []);
    useEffect(()=>{
        updateText();
    }, [
        updateText
    ]);
    return [
        text,
        copy
    ];
};

function getPlatform(userAgent) {
    if (/iPad|iPhone|iPod|ios/i.test(userAgent)) {
        return 'ios';
    } else if (/android/i.test(userAgent)) {
        return 'android';
    } else {
        return 'unknown';
    }
}
const usePlatform = ({ userAgent } = {
    userAgent: ''
})=>{
    const [ua, setUa] = useState(userAgent || '');
    const [platform, setPlatform] = useState(()=>{
        if (userAgent) {
            return getPlatform(userAgent);
        }
        return 'unknown';
    });
    useEffect(()=>{
        setPlatform(getPlatform(navigator.userAgent));
        setUa(navigator.userAgent);
    }, []);
    const isInMiniProgram = useCallback(()=>{
        return /miniprogram/i.test(ua);
    }, [
        ua
    ]);
    const isInWechat = useCallback(()=>{
        return /micromessenger/i.test(ua);
    }, [
        ua
    ]);
    const isiPhoneX = useCallback(()=>{
        return /iPhoneX/i.test(ua);
    }, [
        ua
    ]);
    return {
        platform,
        isInMiniProgram,
        isInWechat,
        isiPhoneX
    };
};

function useMobileLandscape() {
    const [isMobileLandscape, setIsMobileLandscape] = useState(false);
    const [orientation] = useOrientation();
    useEffect(()=>{
        const userAgent = window.navigator.userAgent;
        const isMobile = /Mobi|Android｜iphone/i.test(userAgent);
        setIsMobileLandscape(isMobile && orientation.type === 'landscape-primary');
    }, [
        orientation.type
    ]);
    return isMobileLandscape;
}

const useControlled = (value, defaultValue, onChange)=>{
    const [stateValue, setStateValue] = useState(value !== undefined ? value : defaultValue);
    const isControlled = value !== undefined;
    const onChangeRef = useLatest(onChange);
    const setValue = useCallback((newValue)=>{
        if (!isControlled) {
            setStateValue(newValue);
        }
        onChangeRef.current == null ? void 0 : onChangeRef.current.call(onChangeRef, newValue);
    }, [
        isControlled,
        onChangeRef
    ]);
    return [
        isControlled ? value : stateValue,
        setValue
    ];
};

function useDisclosure(props = {}) {
    const { defaultOpen, isOpen: isOpenProp, onClose: onCloseProp, onOpen: onOpenProp, onChange = ()=>{} } = props;
    const onOpenPropRef = useLatest(onOpenProp);
    const onClosePropRef = useLatest(onCloseProp);
    const [isOpen, setIsOpen] = useControlled(isOpenProp, defaultOpen || false, onChange);
    const isControlled = isOpenProp !== undefined;
    const onClose = useCallback(()=>{
        if (!isControlled) {
            setIsOpen(false);
        }
        onClosePropRef.current == null ? void 0 : onClosePropRef.current.call(onClosePropRef);
    }, [
        isControlled,
        onClosePropRef,
        setIsOpen
    ]);
    const onOpen = useCallback(()=>{
        if (!isControlled) {
            setIsOpen(true);
        }
        onOpenPropRef.current == null ? void 0 : onOpenPropRef.current.call(onOpenPropRef);
    }, [
        isControlled,
        onOpenPropRef,
        setIsOpen
    ]);
    const onOpenChange = useCallback(()=>{
        const action = isOpen ? onClose : onOpen;
        action();
    }, [
        isOpen,
        onOpen,
        onClose
    ]);
    return {
        isOpen: !!isOpen,
        onOpen,
        onClose,
        onOpenChange,
        isControlled
    };
}

const useEventSource = (url, events = [], options = defaultOptions$1)=>{
    const [data, setData] = useState(null);
    const [error, setError] = useState(null);
    const [status, setStatus] = useState('DISCONNECTED');
    const [event, setEvent] = useState(null);
    const [lastEventId, setLastEventId] = useState(null);
    const retries = useRef(0);
    const explicitlyClosed = useRef(false);
    const eventSourceRef = useRef(null);
    const eventListenerRef = useRef();
    if (!eventListenerRef.current) {
        eventListenerRef.current = new Map();
    }
    const clean = useEvent(()=>{
        const listeners = eventListenerRef.current;
        events.forEach((name)=>{
            const handler = listeners == null ? void 0 : listeners.get(name);
            if (handler) {
                var _eventSourceRef_current;
                (_eventSourceRef_current = eventSourceRef.current) == null ? void 0 : _eventSourceRef_current.removeEventListener(name, handler);
            }
        });
    });
    const close = useCallback((explicit = false)=>{
        var _eventSourceRef_current;
        setStatus('DISCONNECTED');
        clean();
        (_eventSourceRef_current = eventSourceRef.current) == null ? void 0 : _eventSourceRef_current.close();
        eventSourceRef.current = null;
        explicitlyClosed.current = explicit;
    }, [
        clean
    ]);
    const explicitlyClose = useCallback(()=>{
        close(true);
    }, [
        close
    ]);
    const open = useEvent(()=>{
        close();
        setStatus('CONNECTING');
        retries.current = 0;
        if (!eventSourceRef.current) {
            eventSourceRef.current = new EventSource(url, {
                withCredentials: options.withCredentials
            });
        }
        const es = eventSourceRef.current;
        es.onopen = ()=>{
            setStatus('CONNECTED');
            setError(null);
        };
        es.onmessage = (ev)=>{
            setData(ev.data);
            setLastEventId(ev.lastEventId);
            setStatus('CONNECTED');
        };
        es.onerror = (err)=>{
            setError(err);
            setStatus('DISCONNECTED');
            if (options.autoReconnect && !explicitlyClosed.current) {
                const { retries: maxRetries = -1, delay = 1000, onFailed } = options.autoReconnect;
                retries.current += 1;
                if (typeof maxRetries === 'number' && (maxRetries < 0 || retries.current < maxRetries) || typeof maxRetries === 'function' && maxRetries()) {
                    setTimeout(open, delay);
                } else {
                    onFailed == null ? void 0 : onFailed();
                }
            }
        };
        const listeners = eventListenerRef.current;
        events.forEach((name)=>{
            const handler = (event)=>{
                setEvent(name);
                var _event_data;
                setData((_event_data = event.data) != null ? _event_data : null);
            };
            es.addEventListener(name, handler);
            listeners == null ? void 0 : listeners.set(name, handler);
        });
    });
    useEffect(()=>{
        if (options.immediate !== false) {
            open();
        }
        return close;
    }, [
        open,
        close,
        options.immediate
    ]);
    useUnmount(()=>{
        close();
    });
    return {
        eventSourceRef,
        data,
        error,
        status,
        lastEventId,
        event,
        close: explicitlyClose,
        open
    };
};

function assignRef(ref, value) {
    if (ref == null) return;
    if (typeof ref === 'function') {
        ref(value);
        return;
    }
    try {
        ref.current = value;
    } catch (error) {
        throw new Error(`Cannot assign value '${value}' to ref '${ref}'`);
    }
}
function mergeRefs(...refs) {
    return (node)=>{
        refs.forEach((ref)=>{
            assignRef(ref, node);
        });
    };
}
function useMergedRefs(...refs) {
    // eslint-disable-next-line react-hooks/exhaustive-deps
    return useMemo(()=>mergeRefs(...refs), refs);
}

/**
 * @description copy from swr
 */ const use = React.use || ((thenable)=>{
    switch(thenable.status){
        case 'pending':
            throw thenable;
        case 'fulfilled':
            return thenable.value;
        case 'rejected':
            throw thenable.reason;
        default:
            thenable.status = 'pending';
            thenable.then((v)=>{
                thenable.status = 'fulfilled';
                thenable.value = v;
            }, (e)=>{
                thenable.status = 'rejected';
                thenable.reason = e;
            });
            throw thenable;
    }
});

function getInitialState(defaultState) {
    // Prevent a React hydration mismatch when a default value is provided by not defaulting to window.matchMedia(query).matches.
    if (defaultState !== undefined) {
        return defaultState;
    }
    if (isBrowser) {
        const navigator1 = window.navigator;
        return navigator1.languages;
    }
    // A default value has not been provided, and you are rendering on the server, warn of a possible hydration mismatch when defaulting to false.
    if (process.env.NODE_ENV !== 'production') {
        console.warn('`usePreferredLanguage` When server side rendering, defaultState should be defined to prevent a hydration mismatches.');
    }
    return [
        'en'
    ];
}
const usePreferredLanguages = (defaultLanguages)=>{
    const [state, setState] = useState(getInitialState(defaultLanguages));
    useEventListener('languagechange', ()=>{
        setState(navigator.languages);
    });
    return state;
};

const useBroadcastChannel = (options)=>{
    const { name } = options;
    const isSupported = useSupported(()=>window && 'BroadcastChannel' in window);
    const [isClosed, setIsClosed] = useState(false);
    const [data, setData] = useState();
    const [error, setError] = useState(null);
    const [timeStamp, setTimeStamp] = useState(0);
    const channelRef = useRef();
    const post = useCallback((data)=>{
        if (channelRef.current) {
            channelRef.current.postMessage(data);
        }
    }, []);
    const close = useCallback(()=>{
        if (channelRef.current) {
            channelRef.current.close();
        }
        setIsClosed(true);
    }, []);
    useEffect(()=>{
        if (isSupported) {
            channelRef.current = new BroadcastChannel(name);
            setError(null);
            const handleMessage = (e)=>{
                setData(e.data);
                // avoid data is same between two messages
                setTimeStamp(Date.now());
            };
            const handleError = (e)=>{
                setError(e);
            };
            const handleClose = ()=>{
                setIsClosed(true);
            };
            channelRef.current.addEventListener('message', handleMessage, {
                passive: true
            });
            channelRef.current.addEventListener('messageerror', handleError, {
                passive: true
            });
            channelRef.current.addEventListener('close', handleClose);
            return ()=>{
                if (channelRef.current) {
                    channelRef.current.removeEventListener('message', handleMessage);
                    channelRef.current.removeEventListener('messageerror', handleError);
                    channelRef.current.removeEventListener('close', handleClose);
                    close();
                }
            };
        }
        return close;
    }, [
        isSupported,
        name,
        close
    ]);
    return {
        isSupported,
        channel: channelRef.current,
        data,
        post,
        close,
        error,
        isClosed,
        timeStamp
    };
};

const useBoolean = (initialValue = false)=>{
    const [value, setValue] = useState(initialValue);
    const setTrue = useEvent(()=>{
        setValue(true);
    });
    const setFalse = useEvent(()=>{
        setValue(false);
    });
    const toggle = useEvent(()=>{
        setValue((prev)=>!prev);
    });
    const handleSetValue = useEvent((newValue)=>{
        setValue(newValue);
    });
    return {
        value,
        setValue: handleSetValue,
        setTrue,
        setFalse,
        toggle
    };
};

const useDevicePixelRatio = ()=>{
    const [pixelRatio, setPixelRatio] = useState(1);
    const observe = useCallback(()=>{
        if (!window) return;
        setPixelRatio(window.devicePixelRatio);
        const media = window.matchMedia(`(resolution: ${window.devicePixelRatio}dppx)`);
        const handleChange = ()=>{
            observe();
        };
        media.addEventListener('change', handleChange, {
            once: true
        });
        return ()=>{
            media.removeEventListener('change', handleChange);
        };
    }, []);
    useEffect(()=>{
        const cleanup = observe();
        return cleanup;
    }, [
        observe
    ]);
    return {
        pixelRatio
    };
};

const useElementByPoint = (options)=>{
    const { x, y, document: doc = typeof document !== 'undefined' ? document : null, multiple = false, interval = 'requestAnimationFrame', immediate = true } = options;
    const isSupported = useSupported(()=>{
        if (multiple) return doc && 'elementsFromPoint' in doc;
        return doc && 'elementFromPoint' in doc;
    });
    const [element, setElement] = useState(null);
    const [isActive, setIsActive] = useState(immediate);
    const rafIdRef = useRef(null);
    const intervalIdRef = useRef(null);
    const getXY = useCallback(()=>{
        // 需要判断 NaN
        const currentX = typeof x === 'function' ? x() : x;
        const currentY = typeof y === 'function' ? y() : y;
        return {
            x: Number.isNaN(currentX) ? 0 : currentX,
            y: Number.isNaN(currentY) ? 0 : currentY
        };
    }, [
        x,
        y
    ]);
    const cb = useCallback(()=>{
        const { x: currentX, y: currentY } = getXY();
        var _doc_elementsFromPoint, _doc_elementFromPoint;
        setElement(multiple ? (_doc_elementsFromPoint = doc == null ? void 0 : doc.elementsFromPoint(currentX, currentY)) != null ? _doc_elementsFromPoint : [] : (_doc_elementFromPoint = doc == null ? void 0 : doc.elementFromPoint(currentX, currentY)) != null ? _doc_elementFromPoint : null);
    }, [
        doc,
        multiple,
        getXY
    ]);
    const cleanup = useCallback(()=>{
        if (rafIdRef.current !== null) {
            cancelAnimationFrame(rafIdRef.current);
            rafIdRef.current = null;
        }
        if (intervalIdRef.current !== null) {
            clearInterval(intervalIdRef.current);
            intervalIdRef.current = null;
        }
    }, []);
    const pause = useCallback(()=>{
        setIsActive(false);
        cleanup();
    }, [
        cleanup
    ]);
    const resume = useCallback(()=>{
        setIsActive(true);
    }, []);
    useEffect(()=>{
        if (!isActive) {
            return;
        }
        if (interval === 'requestAnimationFrame') {
            const runRaf = ()=>{
                cb();
                rafIdRef.current = requestAnimationFrame(runRaf);
            };
            runRaf();
        } else {
            cb();
            intervalIdRef.current = setInterval(cb, interval);
        }
        return cleanup;
    }, [
        isActive,
        interval,
        cb,
        cleanup
    ]);
    useEffect(()=>{
        if (immediate) {
            resume();
        }
        return pause;
    }, [
        immediate,
        resume,
        pause
    ]);
    return {
        isSupported,
        element,
        pause,
        resume,
        isActive
    };
};

function asyncGeneratorStep(gen, resolve, reject, _next, _throw, key, arg) {
    try {
        var info = gen[key](arg);
        var value = info.value;
    } catch (error) {
        reject(error);
        return;
    }
    if (info.done) {
        resolve(value);
    } else {
        Promise.resolve(value).then(_next, _throw);
    }
}
function _async_to_generator(fn) {
    return function() {
        var self = this, args = arguments;
        return new Promise(function(resolve, reject) {
            var gen = fn.apply(self, args);
            function _next(value) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "next", value);
            }
            function _throw(err) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "throw", err);
            }
            _next(undefined);
        });
    };
}
function _extends() {
    _extends = Object.assign || function(target) {
        for(var i = 1; i < arguments.length; i++){
            var source = arguments[i];
            for(var key in source){
                if (Object.prototype.hasOwnProperty.call(source, key)) {
                    target[key] = source[key];
                }
            }
        }
        return target;
    };
    return _extends.apply(this, arguments);
}
function _object_without_properties_loose(source, excluded) {
    if (source == null) return {};
    var target = {};
    var sourceKeys = Object.keys(source);
    var key, i;
    for(i = 0; i < sourceKeys.length; i++){
        key = sourceKeys[i];
        if (excluded.indexOf(key) >= 0) continue;
        target[key] = source[key];
    }
    return target;
}
const useFetchEventSource = (url, options = {})=>{
    const [data, setData] = useState(null);
    const [error, setError] = useState(null);
    const [status, setStatus] = useState('DISCONNECTED');
    const [event, setEvent] = useState(null);
    const [lastEventId, setLastEventId] = useState(null);
    const retries = useRef(0);
    const abortController = useRef(null);
    const explicitlyClosed = useRef(false);
    const close = useEvent(()=>{
        if (!explicitlyClosed.current) {
            var _abortController_current;
            explicitlyClosed.current = true;
            (_abortController_current = abortController.current) == null ? void 0 : _abortController_current.abort();
            abortController.current = null;
            setStatus('DISCONNECTED');
            options.onClose == null ? void 0 : options.onClose.call(options);
        }
    });
    const open = useEvent(()=>_async_to_generator(function*() {
            close();
            setStatus('CONNECTING');
            explicitlyClosed.current = false;
            retries.current = 0;
            // 创建新的 AbortController
            abortController.current = new AbortController();
            try {
                // 从选项中提取 FetchEventSourceInit 相关的选项
                const { immediate, autoReconnect, onOpen, onMessage, onError, onClose, withCredentials, body } = options, fetchOptions = _object_without_properties_loose(options, [
                    "immediate",
                    "autoReconnect",
                    "onOpen",
                    "onMessage",
                    "onError",
                    "onClose",
                    "withCredentials",
                    "body"
                ]);
                // 构建请求配置
                const finalOptions = _extends({
                    method: options.method || 'GET',
                    headers: _extends({
                        'Accept': 'text/event-stream',
                        'Cache-Control': 'no-cache',
                        'Connection': 'keep-alive'
                    }, options.headers),
                    signal: abortController.current.signal,
                    credentials: withCredentials ? 'include' : 'same-origin'
                }, fetchOptions);
                // 只在 POST 请求时添加 body
                if (options.method === 'POST' && body) {
                    finalOptions.body = body;
                    finalOptions.headers = _extends({}, finalOptions.headers, {
                        'Content-Type': 'application/json'
                    });
                }
                yield fetchEventSource(url.toString(), _extends({}, finalOptions, {
                    onopen (response) {
                        return _async_to_generator(function*() {
                            if (response.ok) {
                                setStatus('CONNECTED');
                                setError(null);
                                options.onOpen == null ? void 0 : options.onOpen.call(options);
                            } else {
                                const error = new Error(`Failed to connect: ${response.status} ${response.statusText}`);
                                setError(error);
                                throw error;
                            }
                        })();
                    },
                    onmessage (msg) {
                        if (!explicitlyClosed.current) {
                            setData(msg.data);
                            var _msg_id;
                            setLastEventId((_msg_id = msg.id) != null ? _msg_id : null);
                            setEvent(msg.event || null);
                            options.onMessage == null ? void 0 : options.onMessage.call(options, msg);
                        }
                    },
                    onerror (err) {
                        setError(err);
                        setStatus('DISCONNECTED');
                        const retryDelay = options.onError == null ? void 0 : options.onError.call(options, err);
                        if (options.autoReconnect && !explicitlyClosed.current) {
                            const { retries: maxRetries = -1, delay = 1000, onFailed } = options.autoReconnect;
                            retries.current += 1;
                            if (typeof maxRetries === 'number' && (maxRetries < 0 || retries.current < maxRetries) || typeof maxRetries === 'function' && maxRetries()) {
                                return retryDelay != null ? retryDelay : delay;
                            } else {
                                onFailed == null ? void 0 : onFailed();
                                throw err;
                            }
                        }
                        throw err;
                    },
                    onclose () {
                        if (!explicitlyClosed.current) {
                            setStatus('DISCONNECTED');
                            options.onClose == null ? void 0 : options.onClose.call(options);
                        }
                    }
                }));
            } catch (err) {
                // 只处理非主动关闭导致的错误
                if (!explicitlyClosed.current) {
                    console.error('EventSource Error:', err);
                    setError(err);
                    setStatus('DISCONNECTED');
                }
            }
        })());
    useEffect(()=>{
        if (options.immediate !== false) {
            open();
        }
        return ()=>{
            // 组件卸载时关闭连接
            close();
        };
    }, [
        open,
        close,
        options.immediate
    ]);
    // 组件卸载时确保连接关闭
    useUnmount(()=>{
        close();
    });
    return {
        data,
        error,
        status,
        lastEventId,
        event,
        close,
        open
    };
};

const useMap = (initialValue)=>{
    // avoid exec init code every render
    const initFunc = ()=>{
        if (typeof initialValue === 'function') {
            const result = initialValue();
            return result instanceof Map ? new Map(result) : new Map(result);
        }
        if (initialValue instanceof Map) {
            return new Map(initialValue);
        }
        if (Array.isArray(initialValue)) {
            return new Map(initialValue);
        }
        return new Map();
    };
    const [map, setMap] = useState(initFunc);
    const set = useEvent((key, value)=>{
        setMap((prevMap)=>{
            const newMap = new Map(prevMap);
            newMap.set(key, value);
            return newMap;
        });
    });
    const get = useCallback((key)=>{
        return map.get(key);
    }, [
        map
    ]);
    const remove = useEvent((key)=>{
        const hasKey = map.has(key);
        if (hasKey) {
            setMap((prevMap)=>{
                const newMap = new Map(prevMap);
                newMap.delete(key);
                return newMap;
            });
        }
        return hasKey;
    });
    const has = useCallback((key)=>{
        return map.has(key);
    }, [
        map
    ]);
    const clear = useEvent(()=>{
        setMap(new Map());
    });
    const reset = useEvent(()=>{
        setMap(initFunc());
    });
    return {
        map,
        set,
        get,
        remove,
        has,
        clear,
        reset,
        size: map.size
    };
};

const useSpeechRecognition = (options = {})=>{
    const { interimResults = true, continuous = true, maxAlternatives = 1, lang = 'en-US' } = options;
    const [isListening, setIsListening] = useState(false);
    const [isFinal, setIsFinal] = useState(false);
    const [result, setResult] = useState('');
    const [error, setError] = useState(undefined);
    const recognitionRef = useRef(undefined);
    const SpeechRecognitionClass = defaultWindow && (defaultWindow.SpeechRecognition || defaultWindow.webkitSpeechRecognition);
    const isSupported = useSupported(()=>SpeechRecognitionClass);
    const start = useEvent((startOptions)=>{
        if (!recognitionRef.current) {
            return;
        }
        // Apply options to recognition instance
        const { interimResults: newInterimResults = interimResults, continuous: newContinuous = continuous, maxAlternatives: newMaxAlternatives = maxAlternatives, lang: newLang = lang } = startOptions || {};
        recognitionRef.current.interimResults = newInterimResults;
        recognitionRef.current.continuous = newContinuous;
        recognitionRef.current.maxAlternatives = newMaxAlternatives;
        recognitionRef.current.lang = newLang;
        setIsListening(true);
    });
    const stop = useEvent(()=>{
        setIsListening(false);
    });
    const toggle = useEvent((value = !isListening, startOptions)=>{
        if (value) {
            start(startOptions);
        } else {
            stop();
        }
    });
    // Initialize recognition when supported
    useEffect(()=>{
        if (!isSupported || !SpeechRecognitionClass) {
            return;
        }
        const recognition = new SpeechRecognitionClass();
        recognitionRef.current = recognition;
        // Set up event listeners
        recognition.onstart = ()=>{
            setIsListening(true);
            setIsFinal(false);
        };
        recognition.onresult = (event)=>{
            const currentResult = event.results[event.resultIndex];
            const { transcript } = currentResult[0];
            setIsFinal(currentResult.isFinal);
            setResult(transcript);
            setError(undefined);
        };
        recognition.onerror = (event)=>{
            setError(event);
        };
        recognition.onend = ()=>{
            setIsListening(false);
        };
        return ()=>{
            if (recognition) {
                recognition.abort();
            }
        };
    }, [
        isSupported,
        SpeechRecognitionClass
    ]);
    // Handle listening state changes
    useEffect(()=>{
        if (!recognitionRef.current) {
            return;
        }
        if (isListening) {
            try {
                recognitionRef.current.start();
            } catch (err) {
                console.warn('Failed to start speech recognition:', err);
                setIsListening(false);
            }
        } else {
            try {
                recognitionRef.current.stop();
            } catch (err) {
                console.warn('Failed to stop speech recognition:', err);
            }
        }
    }, [
        isListening
    ]);
    // Cleanup on unmount
    useUnmount(()=>{
        if (recognitionRef.current) {
            recognitionRef.current.abort();
        }
    });
    return {
        isSupported,
        isListening,
        isFinal,
        recognition: recognitionRef.current,
        result,
        error,
        toggle,
        start,
        stop
    };
};

export { assignRef, defaultOptions, mergeRefs, use, useActiveElement, useAsyncEffect, useBoolean, useBroadcastChannel, useClickOutside as useClickAway, useClickOutside, useClipboard, useColorMode, useControlled, useCookie, useClipboard as useCopyToClipboard, useCountDown, useCounter, useCssVar, useCustomCompareEffect, useCycleList, useDarkMode, useDebounce, useDebounceFn, useDeepCompareEffect, useDevicePixelRatio, useDisclosure, useDocumentVisibility, useDoubleClick, useDraggable, useDropZone, useElementBounding, useElementByPoint, useElementSize, useElementVisibility, useEvent, useEventEmitter, useEventListener, useEventSource, useEyeDropper, useFavicon, useFetchEventSource, useFileDialog, useFirstMountState, useFocus, useFps, useFullscreen, useGeolocation, useHover, useIdle, useInfiniteScroll, useIntersectionObserver, useInterval, useIsomorphicLayoutEffect, useKeyModifier, useLatest, useLocalStorage, useLocationSelector, useLongPress, useMap, useMeasure, useMediaDevices, useMediaQuery, useMergedRefs, useMobileLandscape, useMount, useMountedState, useMouse, useMousePressed, useMutationObserver, useNetwork, useObjectUrl, useOnceEffect, useOnceLayoutEffect, useOnline, useOrientation, usePageLeave, usePermission, usePlatform, usePreferredColorScheme, usePreferredContrast, usePreferredDark, usePreferredLanguages, usePrevious, useRafFn, useRafState, useReducedMotion, useResizeObserver, useScratch, useScreenSafeArea, useScriptTag, useScroll, useScrollIntoView, useScrollLock, useSessionStorage, useSetState, useSpeechRecognition, useSticky, useSupported, useTextDirection, useTextSelection, useThrottle, useThrottleFn, useTimeout, useTimeoutFn, useTitle, useToggle, useUnmount, useUpdate, useUpdateEffect, useUpdateLayoutEffect, useWebNotification, useWindowScroll, useWindowSize, useWindowsFocus };
