# CONFIGURATION GUIDE - SEO Fixes Deployment

## How to Complete Your SEO Setup

### Step 1: Get Your Google Analytics 4 ID

1. Go to [Google Analytics](https://analytics.google.com/)
2. Sign in with your Google account
3. Select your property or create a new one for busynepal.com
4. In the left sidebar, click **Admin** → **Property** → **Data Streams**
5. Click on your web data stream
6. Copy your **Measurement ID** (format: G-XXXXXXXXXX)

### Step 2: Update Google Analytics 4 in Your Code

Find and replace in `src/app/layout.tsx`:
- Search for: `G-XXXXXXXXXX`
- Replace ALL occurrences with your actual Measurement ID (e.g., `G-ABC123XYZ`)

There are 2 occurrences to replace:
```tsx
// Line 1: Async script tag
<script async src="https://www.googletagmanager.com/gtag/js?id=G-XXXXXXXXXX"></script>

// Line 2: Configuration
gtag('config', 'G-XXXXXXXXXX', {
```

### Step 3: Get Your Facebook Pixel ID

1. Go to [Facebook Business Manager](https://business.facebook.com/)
2. Navigate to **Events Manager**
3. Select or create your data source
4. Under **Data Source Details**, find your **Pixel ID** (10-15 digit number)
5. Copy the pixel ID

### Step 4: Update Facebook Pixel in Your Code

Find and replace in `src/app/layout.tsx`:
- Search for: `XXXXXXXXXX` (in the Facebook Pixel section)
- Replace with your actual Pixel ID (e.g., `1234567890`)

There are 3 occurrences to replace:
```tsx
// Line 1: In fbq('init')
fbq('init', 'XXXXXXXXXX');

// Line 2: In noscript img src
src="https://www.facebook.com/tr?id=XXXXXXXXXX&ev=PageView&noscript=1"

// Line 3: In noscript img alt (optional, just for reference)
```

### Step 5: Deploy Your Changes

```bash
# 1. Build the project
npm run build

# 2. Start in production
npm run start

# 3. Or deploy to your hosting service
# Follow your hosting provider's deployment process
```

### Step 6: Verify Everything Works

#### Google Analytics:
1. Go to Google Analytics → Real-time reports
2. Open your website in a new browser window
3. You should see yourself as an active user within 5 seconds
4. Check that page_path shows your pages correctly

#### Facebook Pixel:
1. Go to Facebook Business Manager → Events Manager
2. Select your data source (pixel)
3. Open your website in a browser
4. Go to **Test Events** tab
5. You should see a PageView event fire within a few seconds

#### Schema Markup:
1. Go to [Google's Structured Data Testing Tool](https://search.google.com/structured-data/testing-tool)
2. Enter your website URL: https://busynepal.com
3. Verify that Organization and LocalBusiness schemas are detected
4. Look for no errors or warnings

#### Email Protection:
1. Visit your footer or header
2. Try clicking on the email link
3. It should open your default email client

---

## Caddyfile Configuration Verification

The Caddyfile has been updated with:

1. **WWW Redirect**
   - Automatically redirects www.busynepal.com → busynepal.com
   - Test: Visit https://www.busynepal.com in a browser
   - Should redirect to https://busynepal.com

2. **HTTP/2+ Protocol**
   - Already enabled by default in Caddy
   - No additional configuration needed

3. **Test the redirect:**
   ```bash
   # Reload Caddyfile
   killall caddy
   caddy run
   ```

---

## Testing Checklist

- [ ] Google Analytics ID configured
- [ ] Facebook Pixel ID configured
- [ ] Website builds without errors (`npm run build`)
- [ ] Google Analytics showing real-time data
- [ ] Facebook Pixel showing PageView events
- [ ] Schema markup validated (Organization + LocalBusiness)
- [ ] WWW redirect working (www.busynepal.com → busynepal.com)
- [ ] Email links are clickable and work
- [ ] llms.txt file is accessible at https://busynepal.com/llms.txt
- [ ] Robots.txt is accessible at https://busynepal.com/robots.txt

---

## Performance Optimization (Optional but Recommended)

To improve page response time (currently 5.83s, goal: ≤0.8s):

### 1. Enable Image Optimization
Update images to use Next.js Image component:
```tsx
import Image from 'next/image';

// Instead of:
<img src="/busy-post.png" alt="Busy 21" />

// Use:
<Image 
  src="/busy-post.png" 
  alt="Busy 21"
  width={800}
  height={600}
  priority // for above-fold images
/>
```

### 2. Add Caching Headers to Caddyfile
```
header Cache-Control "public, max-age=31536000"
```

### 3. Enable Compression
Add to Caddyfile:
```
encode gzip
```

### 4. Lazy Load Below-Fold Content
```tsx
import dynamic from 'next/dynamic';

const Testimonials = dynamic(() => import('@/components/testimonials'), {
  loading: () => <div>Loading...</div>,
});
```

---

## Troubleshooting

### Google Analytics Not Showing Data
- Check if your GA4 ID is correct
- Check browser console for JavaScript errors
- Allow 24 hours for data to appear in GA4 reports
- Verify you're not blocking Google Analytics with ad blockers

### Facebook Pixel Not Tracking
- Verify your Pixel ID is correct (10-15 digits)
- Check Facebook Events Manager for errors
- Make sure JavaScript is enabled in your browser
- Check browser console for errors

### Schema Markup Not Showing
- Use Google's Structured Data Testing Tool
- Check browser console for JSON errors
- Rebuild and redeploy your site
- Wait 24 hours for Google to re-crawl

### Email Links Not Working
- Ensure the `protected-contact.tsx` component is imported
- Check that the components are used correctly in header/footer
- Test in different browsers

---

## Need Help?

If you encounter any issues:
1. Check the browser console for error messages
2. Review the [SEO_FIXES_SUMMARY.md](./SEO_FIXES_SUMMARY.md) for detailed information
3. Test each component individually
4. Check the official documentation:
   - [Google Analytics Setup](https://support.google.com/analytics/answer/9304153)
   - [Facebook Pixel Guide](https://developers.facebook.com/docs/facebook-pixel/implementation)
   - [Schema.org Documentation](https://schema.org/)
   - [Caddy Documentation](https://caddyserver.com/docs/)

---

**Last Updated:** January 27, 2026
**Website:** busynepal.com
**Company:** Neha Infotech - BusyNepal
