// In-memory storage manager for development
// In production, replace with database (Prisma, MongoDB, etc.)

interface StorageData {
  pages: any[];
  blogPosts: any[];
}

class DataStore {
  private static instance: DataStore;
  private data: StorageData;

  private constructor() {
    this.data = {
      pages: [
        {
          id: '1',
          title: 'About Us',
          slug: 'about',
          content: 'Welcome to our website! Learn more about our team and mission.',
          editorType: 'visual',
          createdAt: new Date(Date.now() - 86400000).toISOString(),
        },
        {
          id: '2',
          title: 'Services',
          slug: 'services',
          content: 'We offer a wide range of professional services to meet your business needs.',
          editorType: 'visual',
          createdAt: new Date(Date.now() - 172800000).toISOString(),
        },
        {
          id: '3',
          title: 'Contact Us',
          slug: 'contact',
          content: 'Get in touch with us for any inquiries or support. We are here to help!',
          editorType: 'visual',
          createdAt: new Date(Date.now() - 259200000).toISOString(),
        },
      ],
      blogPosts: [
        {
          id: '1',
          title: 'Complete Guide to VAT in Nepal with Busy Software',
          slug: 'complete-guide-to-vat-nepal-busy-software',
          excerpt: 'Learn how Busy accounting software can help your business stay compliant with Nepal VAT regulations. From registration to filing, we cover everything.',
          content: 'VAT compliance is crucial for businesses in Nepal. This comprehensive guide covers all aspects of VAT management using Busy accounting software.',
          category: 'VAT Compliance',
          editorType: 'visual',
          createdAt: new Date(Date.now() - 604800000).toISOString(),
        },
        {
          id: '2',
          title: 'Top 10 Benefits of Using Busy Accounting Software',
          slug: 'top-10-benefits-busy-accounting-software',
          excerpt: 'Discover why thousands of businesses in Nepal trust Busy for their accounting needs. From inventory to financial reporting.',
          content: 'Busy accounting software offers numerous benefits for modern businesses. Learn about the top 10 advantages that make it the choice for professionals.',
          category: 'Accounting',
          editorType: 'visual',
          createdAt: new Date(Date.now() - 518400000).toISOString(),
        },
        {
          id: '3',
          title: 'Busy 21 vs Tally: Complete Comparison',
          slug: 'busy-21-vs-tally-comparison',
          excerpt: 'A comprehensive comparison between Busy 21 and Tally accounting software. Make an informed decision for your business.',
          content: 'When choosing accounting software, it is important to understand the differences between popular options like Busy and Tally.',
          category: 'Comparison',
          editorType: 'visual',
          createdAt: new Date(Date.now() - 432000000).toISOString(),
        },
        {
          id: '4',
          title: 'How to Manage Inventory Effectively',
          slug: 'how-to-manage-inventory-effectively',
          excerpt: 'Best practices for inventory management. Reduce stockouts, improve efficiency, and boost profitability.',
          content: 'Effective inventory management is key to business success. Learn the strategies and tools that can help optimize your inventory.',
          category: 'Inventory',
          editorType: 'visual',
          createdAt: new Date(Date.now() - 345600000).toISOString(),
        },
      ],
    };
  }

  static getInstance(): DataStore {
    if (!DataStore.instance) {
      DataStore.instance = new DataStore();
    }
    return DataStore.instance;
  }

  getPages() {
    return this.data.pages;
  }

  getPageById(id: string) {
    return this.data.pages.find((p) => p.id === id);
  }

  createPage(pageData: any) {
    const newPage = {
      id: Date.now().toString(),
      ...pageData,
      createdAt: new Date().toISOString(),
    };
    this.data.pages.push(newPage);
    return newPage;
  }

  updatePage(id: string, pageData: any) {
    const index = this.data.pages.findIndex((p) => p.id === id);
    if (index === -1) return null;

    const updatedPage = {
      ...this.data.pages[index],
      ...pageData,
      id: this.data.pages[index].id,
      createdAt: this.data.pages[index].createdAt,
    };
    this.data.pages[index] = updatedPage;
    return updatedPage;
  }

  deletePage(id: string) {
    const index = this.data.pages.findIndex((p) => p.id === id);
    if (index === -1) return false;

    this.data.pages.splice(index, 1);
    return true;
  }

  getBlogPosts() {
    return this.data.blogPosts;
  }

  getBlogPostById(id: string) {
    return this.data.blogPosts.find((p) => p.id === id);
  }

  createBlogPost(postData: any) {
    const newPost = {
      id: Date.now().toString(),
      ...postData,
      createdAt: new Date().toISOString(),
    };
    this.data.blogPosts.push(newPost);
    return newPost;
  }

  updateBlogPost(id: string, postData: any) {
    const index = this.data.blogPosts.findIndex((p) => p.id === id);
    if (index === -1) return null;

    const updatedPost = {
      ...this.data.blogPosts[index],
      ...postData,
      id: this.data.blogPosts[index].id,
      createdAt: this.data.blogPosts[index].createdAt,
    };
    this.data.blogPosts[index] = updatedPost;
    return updatedPost;
  }

  deleteBlogPost(id: string) {
    const index = this.data.blogPosts.findIndex((p) => p.id === id);
    if (index === -1) return false;

    this.data.blogPosts.splice(index, 1);
    return true;
  }
}

export default DataStore.getInstance();
