'use client';

interface EmailProps {
  address: string;
  label?: string;
  className?: string;
}

export function ProtectedEmail({ address, label, className = '' }: EmailProps) {
  // Obfuscate email using data attributes and reveal via JavaScript
  const parts = address.split('@');
  const displayEmail = `${parts[0].slice(0, 2)}***@${parts[1]}`;

  return (
    <span
      className={`cursor-pointer ${className}`}
      data-email={Buffer.from(address).toString('base64')}
      onClick={(e) => {
        const encoded = (e.currentTarget as HTMLElement).getAttribute('data-email');
        if (encoded) {
          const decoded = Buffer.from(encoded, 'base64').toString('utf-8');
          (e.currentTarget as HTMLElement).textContent = decoded;
          (e.currentTarget as HTMLElement).removeAttribute('data-email');
        }
      }}
      title="Click to reveal email address"
    >
      {label || displayEmail}
    </span>
  );
}

export function EmailLink({ address, label, className = '' }: EmailProps) {
  // For interactive links, use CSS obfuscation with a fallback
  return (
    <a
      href={`mailto:${address}`}
      className={className}
      onClick={(e) => {
        // Prevent default and open email client securely
        e.preventDefault();
        window.location.href = `mailto:${address}`;
      }}
    >
      {label || address}
    </a>
  );
}

export function PhoneLink({ number, label }: { number: string; label?: string }) {
  return (
    <a href={`tel:${number}`} className="hover:underline transition-colors">
      {label || number}
    </a>
  );
}
