'use client';

// CSS inline styles are necessary here for dynamic CSS variables used in the visual page builder
// eslint-disable-next-line jsx-a11y/style-props-do-not-use-inline-styles

import { useState } from 'react';
import './VisualPageBuilder.css';

// Utility to apply CSS variables from data-style attribute
if (typeof window !== 'undefined') {
  document.addEventListener('DOMContentLoaded', () => {
    document.querySelectorAll<HTMLElement>('[data-style]').forEach((el) => {
      const styleString = el.getAttribute('data-style');
      if (styleString) {
        styleString.split(';').forEach((pair) => {
          const [key, value] = pair.split(':').map((s) => s && s.trim());
          if (key && value) {
            el.style.setProperty(key, value);
          }
        });
      }
    });
  });
}
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { 
  Plus, 
  Trash2, 
  Copy, 
  ChevronUp, 
  ChevronDown, 
  Settings2,
  Type,
  Image as ImageIcon,
  Columns,
  File,
  Grid3x3,
  Eye,
  Code
} from 'lucide-react';

interface Block {
  id: string;
  type: 'heading' | 'paragraph' | 'image' | 'button' | 'columns' | 'section' | 'spacer' | 'gallery';
  content: string;
  settings: BlockSettings;
  children?: Block[];
  order: number;
}

interface BlockSettings {
  align?: 'left' | 'center' | 'right';
  textSize?: 'small' | 'medium' | 'large' | 'xl';
  textColor?: string;
  backgroundColor?: string;
  padding?: string;
  margin?: string;
  borderRadius?: string;
  borderColor?: string;
  borderWidth?: string;
  width?: string;
  height?: string;
  customCSS?: string;
}

interface VisualPageBuilderProps {
  initialBlocks?: Block[];
  onSave?: (blocks: Block[], html: string) => void;
}

const BLOCK_TEMPLATES = [
  { id: 'heading', label: 'Heading', icon: Type, description: 'Add a heading' },
  { id: 'paragraph', label: 'Paragraph', icon: File, description: 'Add text content' },
  { id: 'image', label: 'Image', icon: ImageIcon, description: 'Add an image' },
  { id: 'button', label: 'Button', icon: Plus, description: 'Add a button' },
  { id: 'section', label: 'Section', icon: Columns, description: 'Add a section' },
  { id: 'spacer', label: 'Spacer', icon: Grid3x3, description: 'Add spacing' },
];

export const VisualPageBuilder = ({ initialBlocks = [], onSave }: VisualPageBuilderProps) => {
  const [blocks, setBlocks] = useState<Block[]>(
    initialBlocks.length > 0
      ? initialBlocks
      : [
          {
            id: '1',
            type: 'heading',
            content: 'Your Page Title',
            order: 0,
            settings: { textSize: 'xl', align: 'center' },
          },
        ]
  );
  const [selectedBlockId, setSelectedBlockId] = useState<string | null>(null);
  const [previewMode, setPreviewMode] = useState(false);

  const selectedBlock = blocks.find((b) => b.id === selectedBlockId);

  const addBlock = (type: string) => {
    const newBlock: Block = {
      id: Date.now().toString(),
      type: type as Block['type'],
      content: `New ${type}`,
      order: blocks.length,
      settings: {
        padding: '16px',
        margin: '8px 0',
      },
    };
    setBlocks([...blocks, newBlock]);
    setSelectedBlockId(newBlock.id);
  };

  const deleteBlock = (id: string) => {
    setBlocks(blocks.filter((b) => b.id !== id).map((b, idx) => ({ ...b, order: idx })));
    setSelectedBlockId(null);
  };

  const duplicateBlock = (id: string) => {
    const block = blocks.find((b) => b.id === id);
    if (!block) return;
    const newBlock = { ...block, id: Date.now().toString(), order: block.order + 1 };
    setBlocks([...blocks, newBlock]);
  };

  const moveBlock = (id: string, direction: 'up' | 'down') => {
    const index = blocks.findIndex((b) => b.id === id);
    if (index === -1) return;
    if (direction === 'up' && index === 0) return;
    if (direction === 'down' && index === blocks.length - 1) return;

    const newBlocks = [...blocks];
    const newIndex = direction === 'up' ? index - 1 : index + 1;
    [newBlocks[index], newBlocks[newIndex]] = [newBlocks[newIndex], newBlocks[index]];
    newBlocks.forEach((b, idx) => (b.order = idx));
    setBlocks(newBlocks);
  };

  const updateBlockContent = (id: string, content: string) => {
    setBlocks(blocks.map((b) => (b.id === id ? { ...b, content } : b)));
  };

  const updateBlockSettings = (id: string, settings: Partial<BlockSettings>) => {
    setBlocks(
      blocks.map((b) =>
        b.id === id ? { ...b, settings: { ...b.settings, ...settings } } : b
      )
    );
  };

  const getBlockStyleVars = (block: Block): Record<string, string> => ({
    '--vpb-padding': block.settings.padding || '8px',
    '--vpb-margin': block.settings.margin || '4px 0',
    '--vpb-bg-color': block.settings.backgroundColor || 'transparent',
    '--vpb-text-color': block.settings.textColor || 'inherit',
    '--vpb-align': block.settings.align || 'left',
    '--vpb-border-radius': block.settings.borderRadius || '0px',
    '--vpb-width': block.settings.width || '100%',
    '--vpb-height': block.settings.height || 'auto',
    '--vpb-border': block.settings.borderColor
      ? `${block.settings.borderWidth || '1px'} solid ${block.settings.borderColor}`
      : 'none',
    '--vpb-text-size':
      block.settings.textSize === 'small'
        ? '20px'
        : block.settings.textSize === 'medium'
          ? '28px'
          : block.settings.textSize === 'large'
            ? '36px'
            : '48px',
  });

  const renderBlockPreview = (block: Block) => {
    const blockClassName = ['vpb-block', `vpb-block-type-${block.type}`, 'vpb-block-preview'].join(
      ' '
    );
    const blockStyleVars = getBlockStyleVars(block);

    // Generate a style string for CSS variables
    const styleVarString = Object.entries(blockStyleVars)
      .map(([key, value]) => `${key}: ${value}`)
      .join('; ');

    switch (block.type) {
      case 'heading':
        return (
          <h2
            key={block.id}
            className={blockClassName}
            data-style={styleVarString}
          >
            {block.content}
          </h2>
        );
      case 'paragraph':
        return (
          <p
            key={block.id}
            className={blockClassName}
            data-style={styleVarString}
          >
            {block.content}
          </p>
        );
      case 'button':
        return (
          <button
            key={block.id}
            className={blockClassName}
            data-style={styleVarString}
          >
            {block.content}
          </button>
        );
      case 'image':
        return (
          <img
            key={block.id}
            className={blockClassName}
            src={block.content}
            alt={block.content || 'Block image'}
            data-style={styleVarString}
          />
        );
      case 'spacer':
        return (
          <div
            key={block.id}
            className={blockClassName}
            data-style={styleVarString}
          />
        );
      case 'section':
        return (
          <div
            key={block.id}
            className={blockClassName}
            data-style={styleVarString}
          >
            {block.content && <p>{block.content}</p>}
          </div>
        );
      default:
        return null;
    }
  };

  const getBlockIcon = (type: string) => {
    const template = BLOCK_TEMPLATES.find((t) => t.id === type);
    return template?.label || type;
  };

  if (previewMode) {
    return (
      <div className="space-y-4">
        <div className="flex justify-between items-center">
          <h2 className="text-2xl font-bold">Preview</h2>
          <Button onClick={() => setPreviewMode(false)} variant="outline">
            <Settings2 className="w-4 h-4 mr-2" />
            Back to Editor
          </Button>
        </div>
        <Card>
          <CardContent className="pt-6">
            <div className="max-w-4xl mx-auto space-y-0">
              {blocks.map((block) => (
                <div key={block.id}>{renderBlockPreview(block)}</div>
              ))}
            </div>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="grid grid-cols-1 lg:grid-cols-4 gap-6">
      {/* Block Library */}
      <div className="lg:col-span-1">
        <Card>
          <CardHeader>
            <CardTitle className="text-sm">Blocks</CardTitle>
          </CardHeader>
          <CardContent className="space-y-2">
            {BLOCK_TEMPLATES.map((template) => {
              const Icon = template.icon;
              return (
                <Button
                  key={template.id}
                  onClick={() => addBlock(template.id)}
                  variant="outline"
                  className="w-full justify-start gap-2"
                  size="sm"
                >
                  <Icon className="w-4 h-4" />
                  <div className="text-left">
                    <div className="font-medium text-xs">{template.label}</div>
                    <div className="text-xs text-gray-500">{template.description}</div>
                  </div>
                </Button>
              );
            })}
          </CardContent>
        </Card>

        {/* Preview Button */}
        <Button
          onClick={() => setPreviewMode(true)}
          className="w-full mt-4 gap-2"
          variant="default"
        >
          <Eye className="w-4 h-4" />
          Preview
        </Button>
      </div>

      {/* Canvas/Editor */}
      <div className="lg:col-span-2">
        <Card>
          <CardHeader>
            <CardTitle>Page Canvas</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-3 border-l-4 border-blue-500 pl-3 min-h-96 bg-gray-50 p-4 rounded">
              {blocks.length === 0 ? (
                <p className="text-center text-gray-500 py-8">
                  No blocks yet. Add one from the left panel!
                </p>
              ) : (
                blocks.map((block) => (
                  <div
                    key={block.id}
                    onClick={() => setSelectedBlockId(block.id)}
                    onKeyDown={(e) => {
                      if (e.key === 'Enter' || e.key === ' ') {
                        setSelectedBlockId(block.id);
                      }
                    }}
                    role="button"
                    tabIndex={0}
                    className={`vpb-canvas-block ${
                      selectedBlockId === block.id ? 'selected' : ''
                    }`}
                  >
                    <div className="flex items-start justify-between gap-2">
                      <div className="flex-1 min-w-0">
                        <div className="text-xs font-semibold text-gray-600 mb-2">
                          {getBlockIcon(block.type)}
                        </div>
                        <div className="text-sm text-gray-700 truncate">{block.content}</div>
                      </div>
                      <div className="flex gap-1 flex-shrink-0">
                        <Button
                          size="sm"
                          variant="ghost"
                          onClick={(e) => {
                            e.stopPropagation();
                            moveBlock(block.id, 'up');
                          }}
                          aria-label="Move block up"
                        >
                          <ChevronUp className="w-3 h-3" />
                        </Button>
                        <Button
                          size="sm"
                          variant="ghost"
                          onClick={(e) => {
                            e.stopPropagation();
                            moveBlock(block.id, 'down');
                          }}
                          aria-label="Move block down"
                        >
                          <ChevronDown className="w-3 h-3" />
                        </Button>
                        <Button
                          size="sm"
                          variant="ghost"
                          onClick={(e) => {
                            e.stopPropagation();
                            duplicateBlock(block.id);
                          }}
                          aria-label="Duplicate block"
                        >
                          <Copy className="w-3 h-3" />
                        </Button>
                        <Button
                          size="sm"
                          variant="ghost"
                          onClick={(e) => {
                            e.stopPropagation();
                            deleteBlock(block.id);
                          }}
                          aria-label="Delete block"
                        >
                          <Trash2 className="w-3 h-3 text-red-500" />
                        </Button>
                      </div>
                    </div>
                  </div>
                ))
              )}
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Block Settings Panel */}
      <div className="lg:col-span-1">
        {selectedBlock ? (
          <Card>
            <CardHeader>
              <CardTitle className="text-sm">Block Settings</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4 max-h-96 overflow-y-auto">
              {/* Content */}
              <div>
                <label className="block text-xs font-semibold mb-2">Content</label>
                <Textarea
                  value={selectedBlock.content}
                  onChange={(e) => updateBlockContent(selectedBlock.id, e.target.value)}
                  placeholder="Enter block content"
                  rows={3}
                  className="text-xs"
                />
              </div>

              {/* Alignment */}
              <div>
                <label className="block text-xs font-semibold mb-2">Alignment</label>
                <div className="flex gap-2">
                  {['left', 'center', 'right'].map((align) => (
                    <Button
                      key={align}
                      size="sm"
                      variant={selectedBlock.settings.align === align ? 'default' : 'outline'}
                      onClick={() =>
                        updateBlockSettings(selectedBlock.id, {
                          align: align as 'left' | 'center' | 'right',
                        })
                      }
                      className="text-xs flex-1"
                    >
                      {align.charAt(0).toUpperCase() + align.slice(1)}
                    </Button>
                  ))}
                </div>
              </div>

              {/* Text Size */}
              {['heading', 'paragraph'].includes(selectedBlock.type) && (
                <div>
                  <label htmlFor="text-size" className="block text-xs font-semibold mb-2">
                    Text Size
                  </label>
                  <select
                    id="text-size"
                    title="Select text size"
                    value={selectedBlock.settings.textSize || 'medium'}
                    onChange={(e) =>
                      updateBlockSettings(selectedBlock.id, {
                        textSize: e.target.value as any,
                      })
                    }
                    className="w-full px-2 py-1 border rounded text-xs"
                  >
                    <option value="small">Small</option>
                    <option value="medium">Medium</option>
                    <option value="large">Large</option>
                    <option value="xl">Extra Large</option>
                  </select>
                </div>
              )}

              {/* Text Color */}
              <div>
                <label htmlFor="text-color" className="block text-xs font-semibold mb-2">
                  Text Color
                </label>
                <div className="flex gap-2">
                  <input
                    id="text-color"
                    type="color"
                    title="Text color picker"
                    value={selectedBlock.settings.textColor || '#000000'}
                    onChange={(e) =>
                      updateBlockSettings(selectedBlock.id, { textColor: e.target.value })
                    }
                    className="w-10 h-8 rounded border"
                  />
                  <input
                    type="text"
                    placeholder="#000000"
                    value={selectedBlock.settings.textColor || '#000000'}
                    onChange={(e) =>
                      updateBlockSettings(selectedBlock.id, { textColor: e.target.value })
                    }
                    className="flex-1 px-2 py-1 border rounded text-xs"
                  />
                </div>
              </div>

              {/* Background Color */}
              <div>
                <label htmlFor="bg-color" className="block text-xs font-semibold mb-2">
                  Background
                </label>
                <div className="flex gap-2">
                  <input
                    id="bg-color"
                    type="color"
                    title="Background color picker"
                    value={selectedBlock.settings.backgroundColor || '#ffffff'}
                    onChange={(e) =>
                      updateBlockSettings(selectedBlock.id, { backgroundColor: e.target.value })
                    }
                    className="w-10 h-8 rounded border"
                  />
                  <input
                    type="text"
                    placeholder="#ffffff"
                    value={selectedBlock.settings.backgroundColor || '#ffffff'}
                    onChange={(e) =>
                      updateBlockSettings(selectedBlock.id, { backgroundColor: e.target.value })
                    }
                    className="flex-1 px-2 py-1 border rounded text-xs"
                  />
                </div>
              </div>

              {/* Padding */}
              <div>
                <label className="block text-xs font-semibold mb-2">Padding</label>
                <Input
                  value={selectedBlock.settings.padding || '8px'}
                  onChange={(e) =>
                    updateBlockSettings(selectedBlock.id, { padding: e.target.value })
                  }
                  placeholder="e.g., 16px or 10px 20px"
                  className="text-xs"
                />
              </div>

              {/* Margin */}
              <div>
                <label className="block text-xs font-semibold mb-2">Margin</label>
                <Input
                  value={selectedBlock.settings.margin || '8px'}
                  onChange={(e) =>
                    updateBlockSettings(selectedBlock.id, { margin: e.target.value })
                  }
                  placeholder="e.g., 16px or 10px 20px"
                  className="text-xs"
                />
              </div>

              {/* Border Radius */}
              <div>
                <label className="block text-xs font-semibold mb-2">Border Radius</label>
                <Input
                  value={selectedBlock.settings.borderRadius || '0px'}
                  onChange={(e) =>
                    updateBlockSettings(selectedBlock.id, { borderRadius: e.target.value })
                  }
                  placeholder="e.g., 8px"
                  className="text-xs"
                />
              </div>

              {/* Width */}
              <div>
                <label className="block text-xs font-semibold mb-2">Width</label>
                <Input
                  value={selectedBlock.settings.width || '100%'}
                  onChange={(e) =>
                    updateBlockSettings(selectedBlock.id, { width: e.target.value })
                  }
                  placeholder="e.g., 100% or 300px"
                  className="text-xs"
                />
              </div>

              {/* Height */}
              <div>
                <label className="block text-xs font-semibold mb-2">Height</label>
                <Input
                  value={selectedBlock.settings.height || 'auto'}
                  onChange={(e) =>
                    updateBlockSettings(selectedBlock.id, { height: e.target.value })
                  }
                  placeholder="e.g., 300px or auto"
                  className="text-xs"
                />
              </div>
            </CardContent>
          </Card>
        ) : (
          <Card>
            <CardContent className="pt-6 text-center text-gray-500">
              <p className="text-sm">Select a block to edit its settings</p>
            </CardContent>
          </Card>
        )}
      </div>
    </div>
  );
};
