'use client';

import { useState } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { ExternalLink, Loader2 } from 'lucide-react';
import { PageMetadata } from '@/lib/pageRegistry';

interface PagePreviewProps {
  page: PageMetadata;
  onEdit?: (page: PageMetadata) => void;
}

export const RealTimePagePreview = ({ page, onEdit }: PagePreviewProps) => {
  const [loading, setLoading] = useState(false);
  const [iframeLoaded, setIframeLoaded] = useState(false);

  const handleEdit = () => {
    if (onEdit) {
      onEdit(page);
    }
  };

  return (
    <Card className="overflow-hidden h-full flex flex-col">
      <CardHeader>
        <div className="flex items-start justify-between">
          <div className="flex-1">
            <div className="flex items-center gap-2 mb-2">
              <CardTitle>{page.title}</CardTitle>
              <Badge variant={page.type === 'system' ? 'default' : 'secondary'}>
                {page.type === 'system' ? 'System' : 'Custom'}
              </Badge>
            </div>
            <CardDescription className="text-sm">{page.description}</CardDescription>
            <p className="text-xs text-gray-500 mt-2">
              <code className="bg-gray-100 px-2 py-1 rounded">{page.path}</code>
            </p>
          </div>
          <div className="flex gap-2 flex-shrink-0">
            <a
              href={page.path}
              target="_blank"
              rel="noopener noreferrer"
              className="inline-flex"
            >
              <Button size="sm" variant="outline" className="gap-1">
                <ExternalLink className="w-4 h-4" />
                View
              </Button>
            </a>
            {page.type === 'custom' && (
              <Button size="sm" onClick={handleEdit}>
                Edit
              </Button>
            )}
          </div>
        </div>
      </CardHeader>

      <CardContent className="flex-1">
        <div className="relative w-full h-96 border border-gray-200 rounded-lg bg-gray-50 overflow-hidden">
          {!iframeLoaded && (
            <div className="absolute inset-0 flex items-center justify-center bg-white z-10">
              <Loader2 className="w-6 h-6 animate-spin text-gray-400" />
            </div>
          )}
          <iframe
            src={page.path}
            title={page.title}
            className="w-full h-full border-0"
            onLoad={() => setIframeLoaded(true)}
          />
        </div>
        <p className="text-xs text-gray-500 mt-3">
          ℹ️ Real-time preview of the live page. Click "View" to see full page.
        </p>
      </CardContent>
    </Card>
  );
};
