'use client';

import { useState } from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { X, Loader2, Code, Layout, Eye } from 'lucide-react';
import { PagePreview } from './PagePreview';
import { PageBuilder } from './PageBuilder';

interface PageCreateModalProps {
  isOpen: boolean;
  onClose: () => void;
  onCreate: (pageData: {
    title: string;
    slug: string;
    content: string;
    editorType: 'visual' | 'code' | 'builder';
  }) => void;
  editData?: {
    id: string;
    title: string;
    slug: string;
    content: string;
    editorType: 'visual' | 'code' | 'builder';
  } | null;
}

export const PageCreateModal = ({ isOpen, onClose, onCreate, editData }: PageCreateModalProps) => {
  const [title, setTitle] = useState(editData?.title || '');
  const [slug, setSlug] = useState(editData?.slug || '');
  const [content, setContent] = useState(editData?.content || '');
  const [editorType, setEditorType] = useState<'visual' | 'code' | 'builder'>(editData?.editorType || 'visual');
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [activeTab, setActiveTab] = useState('editor');

  const handleTitleChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const value = e.target.value;
    setTitle(value);
    setSlug(value.toLowerCase().replace(/\s+/g, '-'));
  };

  const handleBuilderChange = (sections: any[]) => {
    // Convert sections array to formatted content
    const formattedContent = sections
      .map((section) => {
        switch (section.type) {
          case 'heading':
            return `# ${section.content}`;
          case 'paragraph':
            return section.content;
          case 'button':
            return `[${section.content}]()`;
          case 'divider':
            return '---';
          case 'image':
            return `![${section.content}](image-url)`;
          default:
            return section.content;
        }
      })
      .join('\n\n');
    setContent(formattedContent);
  };

  const handleSubmit = async () => {
    if (!title.trim() || !slug.trim() || !content.trim()) {
      alert('Please fill in all fields');
      return;
    }

    setIsSubmitting(true);
    try {
      onCreate({
        title,
        slug,
        content,
        editorType,
      });
      setTitle('');
      setSlug('');
      setContent('');
      setEditorType('visual');
      setActiveTab('editor');
      onClose();
    } finally {
      setIsSubmitting(false);
    }
  };

  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center p-4">
      <Card className="w-full max-w-4xl max-h-[95vh] overflow-y-auto">
        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-4">
          <div>
            <CardTitle>{editData ? 'Edit Page' : 'Create New Page'}</CardTitle>
            <CardDescription>{editData ? 'Update your page' : 'Add a new page to your website'}</CardDescription>
          </div>
          <button
            onClick={onClose}
            className="text-gray-500 hover:text-gray-700"
            title="Close modal"
            aria-label="Close modal"
          >
            <X className="w-5 h-5" />
          </button>
        </CardHeader>

        <CardContent className="space-y-6">
          {/* Page Info */}
          <div className="space-y-4">
            <div>
              <label className="block text-sm font-medium mb-2">Page Title</label>
              <input
                type="text"
                value={title}
                onChange={handleTitleChange}
                placeholder="Enter page title"
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
              />
            </div>
            <div>
              <label className="block text-sm font-medium mb-2">URL Slug</label>
              <input
                type="text"
                value={slug}
                onChange={(e) => setSlug(e.target.value)}
                placeholder="page-url-slug"
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
              />
            </div>
          </div>

          {/* Editor Tabs */}
          <Tabs value={activeTab} onValueChange={setActiveTab}>
            <TabsList className="grid w-full grid-cols-3">
              <TabsTrigger value="editor" className="gap-2">
                <Layout className="w-4 h-4" />
                Editor
              </TabsTrigger>
              <TabsTrigger value="builder" className="gap-2">
                <Code className="w-4 h-4" />
                Builder
              </TabsTrigger>
              <TabsTrigger value="preview" className="gap-2">
                <Eye className="w-4 h-4" />
                Preview
              </TabsTrigger>
            </TabsList>

            {/* Code/Visual Editor Tab */}
            <TabsContent value="editor" className="space-y-4">
              <div className="flex gap-2 mb-4">
                <button
                  onClick={() => setEditorType('visual')}
                  className={`px-3 py-1 rounded text-sm font-medium transition-colors ${
                    editorType === 'visual'
                      ? 'bg-blue-600 text-white'
                      : 'bg-gray-200 text-gray-700 hover:bg-gray-300'
                  }`}
                >
                  🎨 Visual Editor
                </button>
                <button
                  onClick={() => setEditorType('code')}
                  className={`px-3 py-1 rounded text-sm font-medium transition-colors ${
                    editorType === 'code'
                      ? 'bg-blue-600 text-white'
                      : 'bg-gray-200 text-gray-700 hover:bg-gray-300'
                  }`}
                >
                  💻 Code Editor
                </button>
              </div>

              <div>
                <label className="block text-sm font-medium mb-2">Page Content</label>
                <textarea
                  value={content}
                  onChange={(e) => setContent(e.target.value)}
                  placeholder="Enter your page content here..."
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 min-h-[300px] font-sans"
                />
                <p className="text-xs text-gray-500 mt-2">
                  {editorType === 'visual'
                    ? 'Supports plain text, markdown, and basic HTML'
                    : 'Write valid React/JSX code'}
                </p>
              </div>
            </TabsContent>

            {/* Page Builder Tab */}
            <TabsContent value="builder" className="space-y-4">
              <PageBuilder onChange={handleBuilderChange} />
            </TabsContent>

            {/* Preview Tab */}
            <TabsContent value="preview" className="space-y-4">
              <PagePreview title={title} slug={slug} content={content} editorType={editorType} />
            </TabsContent>
          </Tabs>

          {/* Actions */}
          <div className="flex gap-3 justify-end pt-4 border-t">
            <Button variant="outline" onClick={onClose}>
              Cancel
            </Button>
            <Button onClick={handleSubmit} disabled={isSubmitting}>
              {isSubmitting && <Loader2 className="w-4 h-4 mr-2 animate-spin" />}
              {editData ? 'Update Page' : 'Create Page'}
            </Button>
          </div>
        </CardContent>
      </Card>
    </div>
  );
};
