'use client';

import { useState } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { GripVertical, Trash2, Plus } from 'lucide-react';

interface PageSection {
  id: string;
  type: 'heading' | 'paragraph' | 'image' | 'button' | 'divider';
  content: string;
  order: number;
}

interface PageBuilderProps {
  initialSections?: PageSection[];
  onChange: (sections: PageSection[]) => void;
}

const SECTION_TYPES = [
  { id: 'heading', label: 'Heading', icon: 'H1' },
  { id: 'paragraph', label: 'Paragraph', icon: '¶' },
  { id: 'image', label: 'Image', icon: '🖼️' },
  { id: 'button', label: 'Button', icon: '🔘' },
  { id: 'divider', label: 'Divider', icon: '─' },
];

export const PageBuilder = ({ initialSections = [], onChange }: PageBuilderProps) => {
  const [sections, setSections] = useState<PageSection[]>(
    initialSections.length > 0
      ? initialSections
      : [
          {
            id: '1',
            type: 'heading',
            content: 'Page Title',
            order: 0,
          },
        ]
  );
  const [draggedItem, setDraggedItem] = useState<string | null>(null);

  const handleAddSection = (type: string) => {
    const newSection: PageSection = {
      id: Date.now().toString(),
      type: type as PageSection['type'],
      content: `New ${type}`,
      order: sections.length,
    };
    const newSections = [...sections, newSection];
    setSections(newSections);
    onChange(newSections);
  };

  const handleDeleteSection = (id: string) => {
    const newSections = sections.filter((s) => s.id !== id).map((s, idx) => ({ ...s, order: idx }));
    setSections(newSections);
    onChange(newSections);
  };

  const handleUpdateSection = (id: string, content: string) => {
    const newSections = sections.map((s) => (s.id === id ? { ...s, content } : s));
    setSections(newSections);
    onChange(newSections);
  };

  const handleDragStart = (id: string) => {
    setDraggedItem(id);
  };

  const handleDragOver = (e: React.DragEvent) => {
    e.preventDefault();
  };

  const handleDrop = (targetId: string) => {
    if (!draggedItem || draggedItem === targetId) return;

    const draggedIndex = sections.findIndex((s) => s.id === draggedItem);
    const targetIndex = sections.findIndex((s) => s.id === targetId);

    const newSections = [...sections];
    [newSections[draggedIndex], newSections[targetIndex]] = [newSections[targetIndex], newSections[draggedIndex]];

    newSections.forEach((s, idx) => (s.order = idx));
    setSections(newSections);
    onChange(newSections);
    setDraggedItem(null);
  };

  const renderSection = (section: PageSection) => {
    switch (section.type) {
      case 'heading':
        return <h2 className="text-2xl font-bold">{section.content}</h2>;
      case 'paragraph':
        return <p className="text-base text-gray-700">{section.content}</p>;
      case 'button':
        return <button className="px-4 py-2 bg-blue-600 text-white rounded">{section.content}</button>;
      case 'divider':
        return <hr className="border-gray-300" />;
      case 'image':
        return <div className="bg-gray-200 h-32 flex items-center justify-center text-gray-500">{section.content}</div>;
      default:
        return null;
    }
  };

  return (
    <div className="space-y-6">
      {/* Add Sections Toolbar */}
      <Card>
        <CardHeader>
          <CardTitle className="text-sm">Add Sections</CardTitle>
          <CardDescription>Drag and drop sections to reorder</CardDescription>
        </CardHeader>
        <CardContent>
          <div className="flex flex-wrap gap-2">
            {SECTION_TYPES.map((sectionType) => (
              <Button
                key={sectionType.id}
                onClick={() => handleAddSection(sectionType.id)}
                variant="outline"
                size="sm"
                className="gap-1"
              >
                <Plus className="w-3 h-3" />
                {sectionType.label}
              </Button>
            ))}
          </div>
        </CardContent>
      </Card>

      {/* Page Preview */}
      <Card>
        <CardHeader>
          <CardTitle className="text-sm">Page Preview</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="bg-white rounded border border-gray-200 p-6 space-y-4 min-h-96">
            {sections.map((section) => (
              <div key={section.id} className="hover:bg-gray-50 transition-colors">
                {renderSection(section)}
              </div>
            ))}
          </div>
        </CardContent>
      </Card>

      {/* Sections Editor */}
      <Card>
        <CardHeader>
          <CardTitle className="text-sm">Edit Sections</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="space-y-3 max-h-96 overflow-y-auto">
            {sections.length === 0 ? (
              <p className="text-center text-gray-500 py-8">No sections yet. Add one to get started!</p>
            ) : (
              sections.map((section) => (
                <div
                  key={section.id}
                  draggable
                  onDragStart={() => handleDragStart(section.id)}
                  onDragOver={handleDragOver}
                  onDrop={() => handleDrop(section.id)}
                  className="flex items-start gap-3 p-3 border border-gray-200 rounded-lg bg-gray-50 hover:bg-gray-100 transition-colors cursor-move"
                >
                  <GripVertical className="w-4 h-4 text-gray-400 mt-1 flex-shrink-0" />
                  <div className="flex-1 min-w-0">
                    <div className="flex items-center gap-2 mb-2">
                      <span className="text-xs font-semibold text-gray-600 uppercase">
                        {section.type}
                      </span>
                    </div>
                    <textarea
                      value={section.content}
                      onChange={(e) => handleUpdateSection(section.id, e.target.value)}
                      className="w-full px-2 py-1 border border-gray-300 rounded text-sm font-mono resize-none"
                      rows={2}
                      placeholder="Enter content..."
                    />
                  </div>
                  <button
                    type="button"
                    aria-label="Delete section"
                    onClick={() => handleDeleteSection(section.id)}
                    className="text-red-600 hover:text-red-800 flex-shrink-0 mt-1"
                  >
                    <Trash2 className="w-4 h-4" />
                  </button>
                </div>
              ))
            )}
          </div>
        </CardContent>
      </Card>
    </div>
  );
};
