'use client';

import { useState } from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Eye, Edit, Trash2, Plus, Loader2, Calendar, Clock } from 'lucide-react';

interface BlogPost {
  id: string;
  title: string;
  slug: string;
  excerpt: string;
  content: string;
  category: string;
  createdAt: string;
  editorType: 'visual' | 'code';
}

interface BlogTabProps {
  posts: BlogPost[];
  loading: boolean;
  onCreateNew: () => void;
  onEdit: (post: BlogPost) => void;
  onDelete: (postId: string) => void;
}

const CATEGORY_COLORS: { [key: string]: string } = {
  Accounting: 'bg-blue-100 text-blue-800',
  'VAT Compliance': 'bg-purple-100 text-purple-800',
  Inventory: 'bg-green-100 text-green-800',
  Reports: 'bg-orange-100 text-orange-800',
  Migration: 'bg-red-100 text-red-800',
  Comparison: 'bg-pink-100 text-pink-800',
};

export const BlogTab = ({
  posts,
  loading,
  onCreateNew,
  onEdit,
  onDelete,
}: BlogTabProps) => {
  const [selectedPost, setSelectedPost] = useState<BlogPost | null>(null);
  const [deleting, setDeleting] = useState<string | null>(null);

  const handleDelete = async (postId: string) => {
    if (!confirm('Are you sure you want to delete this post?')) return;

    setDeleting(postId);
    try {
      await onDelete(postId);
      if (selectedPost?.id === postId) {
        setSelectedPost(null);
      }
    } finally {
      setDeleting(null);
    }
  };

  const getReadTime = (content: string) => {
    const wordsPerMinute = 200;
    const wordCount = content.split(/\s+/).length;
    return Math.ceil(wordCount / wordsPerMinute);
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex justify-between items-center">
        <div>
          <h2 className="text-2xl font-bold">Blog Management</h2>
          <p className="text-gray-600 text-sm">Create and manage blog posts</p>
        </div>
        <Button onClick={onCreateNew} className="gap-2">
          <Plus className="w-4 h-4" />
          New Post
        </Button>
      </div>

      {/* Content Grid */}
      <div className="grid lg:grid-cols-3 gap-8">
        {/* Posts List */}
        <div className="lg:col-span-2">
          <Card>
            <CardHeader>
              <CardTitle>Blog Posts</CardTitle>
              <CardDescription>Click on a post to view details and manage it</CardDescription>
            </CardHeader>
            <CardContent>
              {loading ? (
                <div className="flex items-center justify-center py-8">
                  <Loader2 className="w-6 h-6 animate-spin text-gray-400" />
                </div>
              ) : posts.length === 0 ? (
                <div className="text-center py-8">
                  <p className="text-gray-500 mb-4">No blog posts yet</p>
                  <Button onClick={onCreateNew} size="sm">
                    Create Your First Post
                  </Button>
                </div>
              ) : (
                <div className="space-y-3">
                  {posts.map((post) => (
                    <div
                      key={post.id}
                      onClick={() => setSelectedPost(post)}
                      className={`p-4 border rounded-lg cursor-pointer transition-all ${
                        selectedPost?.id === post.id
                          ? 'border-purple-500 bg-purple-50'
                          : 'border-gray-200 hover:border-gray-300'
                      }`}
                    >
                      <div className="flex items-start justify-between">
                        <div className="flex-1">
                          <h3 className="font-semibold">{post.title}</h3>
                          <p className="text-sm text-gray-600 line-clamp-2">{post.excerpt}</p>
                          <div className="flex items-center gap-2 mt-2 flex-wrap">
                            <Badge
                              className={`text-xs ${
                                CATEGORY_COLORS[post.category] || 'bg-gray-100 text-gray-800'
                              }`}
                            >
                              {post.category}
                            </Badge>
                            <span className="text-xs text-gray-500">
                              {new Date(post.createdAt).toLocaleDateString()}
                            </span>
                            <Badge variant="outline" className="text-xs">
                              {post.editorType === 'visual' ? '🎨 Visual' : '💻 Code'}
                            </Badge>
                          </div>
                        </div>
                        <Eye className="w-4 h-4 text-gray-400 flex-shrink-0 mt-1" />
                      </div>
                    </div>
                  ))}
                </div>
              )}
            </CardContent>
          </Card>
        </div>

        {/* Details Panel */}
        <div>
          {selectedPost ? (
            <Card>
              <CardHeader>
                <CardTitle className="text-lg">Post Details</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <label className="text-xs font-semibold text-gray-500 uppercase">Title</label>
                  <p className="text-sm font-medium line-clamp-2">{selectedPost.title}</p>
                </div>

                <div>
                  <label className="text-xs font-semibold text-gray-500 uppercase">URL Slug</label>
                  <p className="text-sm text-gray-700 font-mono">/{selectedPost.slug}</p>
                </div>

                <div>
                  <label className="text-xs font-semibold text-gray-500 uppercase">Category</label>
                  <p className="text-sm">
                    <Badge
                      className={`text-xs ${
                        CATEGORY_COLORS[selectedPost.category] || 'bg-gray-100 text-gray-800'
                      }`}
                    >
                      {selectedPost.category}
                    </Badge>
                  </p>
                </div>

                <div>
                  <label className="text-xs font-semibold text-gray-500 uppercase">Excerpt</label>
                  <p className="text-sm text-gray-700 line-clamp-3">{selectedPost.excerpt}</p>
                </div>

                <div className="grid grid-cols-2 gap-3">
                  <div>
                    <label className="text-xs font-semibold text-gray-500 uppercase">Published</label>
                    <p className="text-sm text-gray-600 flex items-center gap-1">
                      <Calendar className="w-3 h-3" />
                      {new Date(selectedPost.createdAt).toLocaleDateString()}
                    </p>
                  </div>
                  <div>
                    <label className="text-xs font-semibold text-gray-500 uppercase">Read Time</label>
                    <p className="text-sm text-gray-600 flex items-center gap-1">
                      <Clock className="w-3 h-3" />
                      {getReadTime(selectedPost.excerpt)} min
                    </p>
                  </div>
                </div>

                <div className="flex flex-col gap-2 pt-4 border-t">
                  <Button
                    onClick={() => onEdit(selectedPost)}
                    variant="outline"
                    className="w-full gap-2"
                  >
                    <Edit className="w-4 h-4" />
                    Edit Post
                  </Button>
                  <Button
                    onClick={() => handleDelete(selectedPost.id)}
                    disabled={deleting === selectedPost.id}
                    variant="destructive"
                    className="w-full"
                  >
                    {deleting === selectedPost.id ? (
                      <>
                        <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                        Deleting...
                      </>
                    ) : (
                      <>
                        <Trash2 className="w-4 h-4 mr-2" />
                        Delete Post
                      </>
                    )}
                  </Button>
                </div>
              </CardContent>
            </Card>
          ) : (
            <Card>
              <CardContent className="py-12">
                <p className="text-center text-gray-500">
                  Select a post to view details
                </p>
              </CardContent>
            </Card>
          )}
        </div>
      </div>
    </div>
  );
};
