'use client';

import { useState } from 'react';
import { Card, CardContent } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Badge } from '@/components/ui/badge';
import { RealTimePagePreview } from './RealTimePagePreview';
import { PageBuilder } from './PageBuilder';
import { PagePreview } from './PagePreview';
import { VisualPageBuilder } from './VisualPageBuilder';
import { PageMetadata, pageRegistry, getSystemPages, getCustomPages } from '@/lib/pageRegistry';
import { ExternalLink, Edit2, Save, X, Plus } from 'lucide-react';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';

interface AllPagesViewProps {
  onEditPage?: (page: PageMetadata) => void;
  onCreatePage?: () => void;
}

export const AllPagesView = ({ onEditPage, onCreatePage }: AllPagesViewProps) => {
  const [selectedPage, setSelectedPage] = useState<PageMetadata | null>(null);
  const [editingPageId, setEditingPageId] = useState<string | null>(null);
  const [editingContent, setEditingContent] = useState('');
  const [editingTitle, setEditingTitle] = useState('');
  const [editingSlug, setEditingSlug] = useState('');
  const [editingDescription, setEditingDescription] = useState('');
  const [editorMode, setEditorMode] = useState<'visual' | 'code' | 'builder'>('visual');
  const [sections, setSections] = useState<any[]>([]);
  const systemPages = getSystemPages();
  const customPages = getCustomPages();

  const handleSelectPage = (page: PageMetadata) => {
    setSelectedPage(page);
    setEditingPageId(null);
  };

  const handleStartEdit = (page: PageMetadata) => {
    setSelectedPage(page);
    setEditingPageId(page.id);
    setEditingTitle(page.title);
    setEditingSlug(page.slug);
    setEditingDescription(page.description);
    setEditingContent(page.content);
    setEditorMode(page.editorType);
    
    // For builder mode, try to reconstruct sections from stored content
    if (page.editorType === 'builder' && page.content) {
      try {
        // If content is JSON, parse it as sections
        const parsed = JSON.parse(page.content);
        if (Array.isArray(parsed)) {
          setSections(parsed);
        } else {
          setSections([]);
        }
      } catch {
        // If not JSON, leave sections empty
        setSections([]);
      }
    } else {
      setSections([]);
    }
    
    // Scroll to the edit section
    setTimeout(() => {
      document.querySelector('[data-edit-section]')?.scrollIntoView({ behavior: 'smooth' });
    }, 100);
  };

  const handleSaveEdit = async () => {
    if (!editingPageId || !selectedPage) return;

    try {
      const url = `/api/admin/pages/${editingPageId}`;
      const response = await fetch(url, {
        method: 'PUT',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          id: editingPageId,
          title: editingTitle,
          slug: editingSlug,
          content: editingContent,
          editorType: editorMode,
        }),
      });

      if (response.ok) {
        const updated = await response.json();
        setSelectedPage(updated);
        setEditingPageId(null);
        alert('Page updated successfully!');
      } else {
        alert('Failed to update page');
      }
    } catch (error) {
      console.error('Error saving page:', error);
      alert('Error saving page');
    }
  };

  const handleCancelEdit = () => {
    setEditingPageId(null);
    setEditingContent('');
    setEditingTitle('');
    setEditingSlug('');
    setEditingDescription('');
  };

  const handleBuilderChange = (newSections: any[]) => {
    setSections(newSections);
    // Convert sections to markdown format
    const markdown = newSections
      .map((s) => {
        switch (s.type) {
          case 'heading':
            return `# ${s.content}`;
          case 'paragraph':
            return s.content;
          case 'button':
            return `[${s.content}](#)`;
          case 'divider':
            return '---';
          case 'image':
            return `![Image](${s.content})`;
          default:
            return s.content;
        }
      })
      .join('\n\n');
    setEditingContent(markdown);
  };

  return (
    <div className="space-y-6">
      {/* Header with Create Button */}
      <div className="flex justify-between items-center">
        <div>
          <h2 className="text-2xl font-bold">Pages Management</h2>
          <p className="text-gray-600 text-sm mt-1">Manage all system and custom pages</p>
        </div>
        <Button onClick={onCreatePage} className="gap-2">
          <Plus className="w-4 h-4" />
          Create New Page
        </Button>
      </div>

      {/* Pages Tabs */}
      <Tabs defaultValue="system" className="w-full">
        <TabsList className="grid w-full max-w-md grid-cols-2">
          <TabsTrigger value="system">
            System Pages ({systemPages.length})
          </TabsTrigger>
          <TabsTrigger value="custom">
            Custom Pages ({customPages.length})
          </TabsTrigger>
        </TabsList>

        <TabsContent value="system" className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
            {systemPages.map((page) => (
              <Card
                key={page.id}
                className={`cursor-pointer transition-all hover:shadow-lg ${
                  selectedPage?.id === page.id ? 'ring-2 ring-blue-500' : ''
                }`}
                onClick={() => handleSelectPage(page)}
              >
                <CardContent className="pt-6">
                  <div className="space-y-3">
                    <div>
                      <h3 className="font-semibold text-lg">{page.title}</h3>
                      <p className="text-sm text-gray-600 line-clamp-2">{page.description}</p>
                    </div>
                    <div className="space-y-2">
                      <code className="text-xs bg-gray-100 p-1 rounded block text-gray-700">
                        {page.path}
                      </code>
                      <Badge variant="secondary" className="block w-fit">
                        System Page
                      </Badge>
                    </div>
                      <div className="flex gap-2">
                        <a
                          href={page.path}
                          target="_blank"
                          rel="noopener noreferrer"
                          onClick={(e) => e.stopPropagation()}
                          className="flex-1"
                        >
                          <Button size="sm" variant="outline" className="w-full gap-1">
                            <ExternalLink className="w-3 h-3" />
                            View
                          </Button>
                        </a>
                        <Button
                          size="sm"
                          className="flex-1 gap-1"
                          onClick={(e) => {
                            e.stopPropagation();
                            handleStartEdit(page);
                          }}
                        >
                          <Edit2 className="w-3 h-3" />
                          Modify
                        </Button>
                      </div>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </TabsContent>

        <TabsContent value="custom" className="space-y-4">
          {customPages.length === 0 ? (
            <Card className="bg-blue-50 border-blue-200">
              <CardContent className="pt-6 text-center py-12">
                <p className="text-gray-600 mb-4">No custom pages created yet.</p>
                <Button onClick={() => onEditPage?.({} as PageMetadata)}>
                  Create First Page
                </Button>
              </CardContent>
            </Card>
          ) : (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
              {customPages.map((page) => (
                <Card
                  key={page.id}
                  className={`cursor-pointer transition-all hover:shadow-lg ${
                    selectedPage?.id === page.id ? 'ring-2 ring-blue-500' : ''
                  }`}
                  onClick={() => handleSelectPage(page)}
                >
                  <CardContent className="pt-6">
                    <div className="space-y-3">
                      <div>
                        <h3 className="font-semibold text-lg">{page.title}</h3>
                        <p className="text-sm text-gray-600 line-clamp-2">
                          {page.description || 'No description'}
                        </p>
                      </div>
                      <div className="space-y-2">
                        <code className="text-xs bg-gray-100 p-1 rounded block text-gray-700">
                          {page.slug}
                        </code>
                        <Badge variant="default" className="block w-fit">
                          Custom Page
                        </Badge>
                      </div>
                      <div className="flex gap-2">
                        <a
                          href={`/${page.slug}`}
                          target="_blank"
                          rel="noopener noreferrer"
                          onClick={(e) => e.stopPropagation()}
                          className="flex-1"
                        >
                          <Button size="sm" variant="outline" className="w-full gap-1">
                            <ExternalLink className="w-3 h-3" />
                            View
                          </Button>
                        </a>
                        <Button
                          size="sm"
                          className="flex-1 gap-1"
                          onClick={(e) => {
                            e.stopPropagation();
                            handleStartEdit(page);
                          }}
                        >
                          <Edit2 className="w-3 h-3" />
                          Modify
                        </Button>
                      </div>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          )}
        </TabsContent>
      </Tabs>

      {/* Selected Page View or Edit */}
      {selectedPage && (
        <div className="mt-8 border-t pt-8" data-edit-section>
          <div className="flex items-center justify-between mb-6">
            <h2 className="text-2xl font-bold">{editingPageId ? 'Edit Page' : 'Page Preview'}</h2>
            <div className="flex gap-2">
              {!editingPageId ? (
                <Button onClick={() => handleStartEdit(selectedPage)} className="gap-2">
                  <Edit2 className="w-4 h-4" />
                  Edit Directly
                </Button>
              ) : (
                <>
                  <Button onClick={handleSaveEdit} variant="default" className="gap-2">
                    <Save className="w-4 h-4" />
                    Save Changes
                  </Button>
                  <Button onClick={handleCancelEdit} variant="outline" className="gap-2">
                    <X className="w-4 h-4" />
                    Cancel
                  </Button>
                </>
              )}
            </div>
          </div>

          {/* Edit Mode */}
          {editingPageId && editingPageId !== null && (
            <div className="space-y-6">
              {/* Page Metadata */}
              <Card>
                <CardContent className="pt-6 space-y-4">
                  <div>
                    <label className="block text-sm font-medium mb-1">Page Title</label>
                    <Input
                      value={editingTitle}
                      onChange={(e) => setEditingTitle(e.target.value)}
                      placeholder="Page title"
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium mb-1">Page Slug</label>
                    <Input
                      value={editingSlug}
                      onChange={(e) => setEditingSlug(e.target.value)}
                      placeholder="page-slug"
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium mb-1">Description</label>
                    <Textarea
                      value={editingDescription}
                      onChange={(e) => setEditingDescription(e.target.value)}
                      placeholder="Page description"
                      rows={2}
                    />
                  </div>
                </CardContent>
              </Card>

              {/* Editor Tabs */}
              <Tabs value={editorMode} onValueChange={(v) => setEditorMode(v as 'visual' | 'code' | 'builder')}>
                <TabsList className="grid w-full max-w-md grid-cols-3">
                  <TabsTrigger value="visual">Visual Editor</TabsTrigger>
                  <TabsTrigger value="code">Code Editor</TabsTrigger>
                  <TabsTrigger value="builder">Builder</TabsTrigger>
                </TabsList>

                {/* Visual/Code Editor Tab */}
                <TabsContent value="visual" className="space-y-4">
                  <Card>
                    <CardContent className="pt-6">
                      <label className="block text-sm font-medium mb-2">Page Content</label>
                      <Textarea
                        value={editingContent}
                        onChange={(e) => setEditingContent(e.target.value)}
                        placeholder="Write your page content here..."
                        rows={12}
                        className="font-mono text-sm"
                      />
                    </CardContent>
                  </Card>
                </TabsContent>

                <TabsContent value="code" className="space-y-4">
                  <Card>
                    <CardContent className="pt-6">
                      <label className="block text-sm font-medium mb-2">Page Code</label>
                      <Textarea
                        value={editingContent}
                        onChange={(e) => setEditingContent(e.target.value)}
                        placeholder="Write your JSX/HTML code here..."
                        rows={12}
                        className="font-mono text-sm"
                      />
                    </CardContent>
                  </Card>
                </TabsContent>

                {/* Builder Tab */}
                <TabsContent value="builder" className="space-y-4">
                  <VisualPageBuilder
                    initialBlocks={sections}
                    onSave={(blocks, html) => {
                      handleBuilderChange(blocks);
                    }}
                  />
                </TabsContent>
              </Tabs>

              {/* Live Preview */}
              <Card>
                <CardContent className="pt-6">
                  <h3 className="text-lg font-semibold mb-4">Live Preview</h3>
                  {editorMode === 'builder' ? (
                    <div className="text-sm text-gray-600 p-4 bg-gray-50 rounded">
                      Builder preview is shown in the Builder tab above
                    </div>
                  ) : (
                    <PagePreview
                      title={editingTitle}
                      slug={editingSlug}
                      content={editingContent}
                      editorType={editorMode}
                    />
                  )}
                </CardContent>
              </Card>
            </div>
          )}

          {/* Preview Mode */}
          {!editingPageId && (
            <RealTimePagePreview
              page={selectedPage}
              onEdit={(page) => handleStartEdit(page)}
            />
          )}
        </div>
      )}
    </div>
  );
};
