import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import Image from 'next/image';
import {
  CheckCircle2,
  Cloud,
  Smartphone,
  Shield,
  Database,
  Zap,
  RefreshCw,
  Globe,
  Lock,
  Server,
  Clock,
  FileText,
  Receipt,
  Check,
} from 'lucide-react';
import Link from 'next/link';

export default function BusyOnCloudPage() {
  const features = [
    { icon: Cloud, title: 'Cloud-Based', description: 'Access from anywhere, anytime' },
    { icon: Shield, title: 'Secure', description: 'Bank-grade security with SSL encryption' },
    { icon: RefreshCw, title: 'Auto Backup', description: 'Automatic daily backups' },
    { icon: Globe, title: 'Multi-Location', description: 'Manage multiple locations seamlessly' },
    { icon: Database, title: 'Real-time Sync', description: 'Instant data synchronization' },
  ];

  const benefits = [
    {
      icon: Zap,
      title: 'No Installation Required',
      description: 'Access directly from web browser',
    },
    {
      icon: Lock,
      title: 'Data Security',
      description: 'Enterprise-grade security and compliance',
    },
    {
      icon: Server,
      title: 'No Maintenance',
      description: 'We handle all server maintenance',
    },
    {
      icon: Clock,
      title: '99.9% Uptime',
      description: 'Reliable service with minimal downtime',
    },
  ];

  const plans = [
    {
      id: 'internal',
      name: 'Busy on Cloud (Internal)',
      price: 'NPR 18,000',
      period: 'per year',
      description: 'Perfect for managing internal accounting, inventory, and business operations.',
      highlight: false,
      features: [
        'Busy on Cloud Software Access',
        'Web Access',
        'Automatic Backups',
        'Inventory Management',
        '24/7 Technical Support',
        'Standard Accounting Reports',
      ],
    },
    {
      id: 'ird',
      name: 'Busy on Cloud (IRD)',
      price: 'NPR 25,000',
      period: 'per year',
      description: 'Approved by Inland Revenue Department (IRD)-Nepal. complete solution with integrated IRD/VAT compliance.',
      highlight: true,
      badge: 'Compliant Ready',
      features: [
        'All Internal Features',
        'IRD Approved',
        'E-Invoicing Support',
        'Nepal VAT Reports',
        'Real-time Tax Calculation',
        'Priority 24/7 Support',
      ],
    },
  ];

  return (
    <>
      {/* Hero Section */}
      <section className="bg-gradient-to-br from-purple-500 to-purple-600 text-white py-16 pt-45">
        <div className="container mx-auto px-4">
          <div className="grid lg:grid-cols-2 gap-8 items-center">
            {/* Content */}
            <div>
              <div className="inline-flex items-center gap-2 bg-white/20 backdrop-blur-sm px-4 py-2 rounded-full text-sm mb-4">
                <Cloud className="h-4 w-4" />
                <span>Future of Accounting is Here</span>
              </div>
              <h1 className="text-4xl lg:text-5xl font-bold mb-4">
                Busy on Cloud
              </h1>
              <p className="text-xl text-purple-100 mb-6">
                Access your business data from anywhere, anytime. Choose between standard internal
                management or full IRD compliance.
              </p>
              <div className="flex flex-wrap gap-4">
                <Link href="/downloads">
                  <Button size="lg" className="bg-white text-purple-600 hover:bg-purple-50">
                    Start Free Trial
                  </Button>
                </Link>
                <Link href="/contact">
                  <Button size="lg" variant="outline" className="border-white text-white hover:bg-white hover:text-purple-600">
                    Contact Sales
                  </Button>
                </Link>
              </div>
            </div>
            
            {/* Image */}
            <div className="flex items-center justify-center">
              <div className="relative w-full max-w-md aspect-square bg-white/10 rounded-2xl border-2 border-white/20 overflow-hidden">
                <Image
                  src="/Busyoncloud.png"
                  alt="Busy on Cloud Product"
                  fill
                  className="object-cover"
                  priority
                />
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Scrolling Announcement Banner */}
      <style>{`
        @keyframes scroll-left {
          0% {
            transform: translateX(100%);
          }
          100% {
            transform: translateX(-100%);
          }
        }
        .scrolling-banner {
          overflow: hidden;
          white-space: nowrap;
        }
        .scroll-text {
          display: inline-block;
          animation: scroll-left 15s linear infinite;
          padding: 0 20px;
          font-size: 1.25rem;
          font-weight: 700;
          letter-spacing: 1px;
          background: linear-gradient(90deg, rgba(255, 215, 0, 0.5), rgba(255, 255, 255, 0.3), rgba(255, 215, 0, 0.5));
          padding: 12px 24px;
          border-radius: 50px;
          border: 2px solid rgba(255, 215, 0, 0.6);
          box-shadow: 0 0 20px rgba(255, 215, 0, 0.4);
        }
      `}</style>
      <div className="bg-gradient-to-r from-purple-600 to-purple-700 text-white py-6 scrolling-banner">
        <div className="scroll-text">
          🎉 Busy on Cloud(IRD) is Approved in Nepal Now 🎉
        </div>
      </div>

      {/* Features */}
      <section className="py-16 bg-white">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Why Choose Busy on Cloud?
            </h2>
            <p className="text-lg text-gray-600 max-w-2xl mx-auto">
              All the power of Busy Accounting Software with cloud flexibility
            </p>
          </div>
          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
            {features.map((feature, index) => (
              <Card key={index} className="border-2 hover:border-purple-300 transition-all hover:shadow-lg">
                <CardContent className="p-6">
                  <div className="w-12 h-12 bg-purple-100 rounded-xl flex items-center justify-center mb-4">
                    <feature.icon className="h-6 w-6 text-purple-600" />
                  </div>
                  <h3 className="text-lg font-semibold text-gray-900 mb-2">
                    {feature.title}
                  </h3>
                  <p className="text-gray-600 text-sm leading-relaxed">
                    {feature.description}
                  </p>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Benefits */}
      <section className="py-16 bg-gray-50">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Cloud Benefits
            </h2>
            <p className="text-lg text-gray-600 max-w-2xl mx-auto">
              Advantages of moving your accounting to the cloud
            </p>
          </div>
          <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-6">
            {benefits.map((benefit, index) => (
              <Card key={index} className="border-2">
                <CardContent className="p-6 text-center">
                  <benefit.icon className="h-10 w-10 text-purple-600 mx-auto mb-3" />
                  <h3 className="font-semibold text-gray-900 mb-2">
                    {benefit.title}
                  </h3>
                  <p className="text-sm text-gray-600 leading-relaxed">
                    {benefit.description}
                  </p>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* How It Works */}
      <section className="py-16 bg-white">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              How It Works
            </h2>
            <p className="text-lg text-gray-600 max-w-2xl mx-auto">
              Get started in just 3 simple steps
            </p>
          </div>
          <div className="max-w-4xl mx-auto">
            <div className="space-y-8">
              {[
                { step: '1', title: 'Sign Up', description: 'Create your account and choose your plan' },
                { step: '2', title: 'Setup', description: 'Configure your company settings and preferences' },
                { step: '3', title: 'Start Using', description: 'Begin using Busy on Cloud immediately' },
              ].map((item, index) => (
                <div key={index} className="flex gap-6 items-start">
                  <div className="flex-shrink-0">
                    <div className="w-16 h-16 bg-gradient-to-br from-purple-500 to-purple-600 rounded-xl flex items-center justify-center text-white font-bold text-xl">
                      {item.step}
                    </div>
                  </div>
                  <Card className="flex-1 border-2">
                    <CardContent className="p-6">
                      <h3 className="text-lg font-semibold text-gray-900 mb-2">
                        {item.title}
                      </h3>
                      <p className="text-gray-600">{item.description}</p>
                    </CardContent>
                  </Card>
                </div>
              ))}
            </div>
          </div>
        </div>
      </section>

      {/* Pricing Section (Updated for Internal & IRD) */}
      <section className="py-16 bg-gray-50">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Choose Your Plan
            </h2>
            <p className="text-lg text-gray-600">
              Select the best fit for your business compliance needs.
            </p>
          </div>
          
          <div className="max-w-5xl mx-auto grid md:grid-cols-2 gap-8 items-start">
            {plans.map((plan) => (
              <Card 
                key={plan.id} 
                className={`relative flex flex-col h-full border-2 transition-all ${
                  plan.highlight 
                    ? 'border-purple-600 shadow-2xl scale-105 z-10' 
                    : 'border-gray-200 shadow-sm'
                }`}
              >
                {plan.highlight && (
                  <div className="absolute top-0 right-0 bg-purple-600 text-white text-xs font-bold px-3 py-1 rounded-bl-lg">
                    {plan.badge}
                  </div>
                )}
                
                <CardHeader className="pb-4">
                  <div className="flex items-center gap-2 mb-2">
                    {plan.id === 'ird' ? <Receipt className="h-6 w-6 text-purple-600" /> : <Database className="h-6 w-6 text-gray-600" />}
                    <CardTitle className="text-xl">{plan.name}</CardTitle>
                  </div>
                  <CardDescription className="text-sm min-h-[40px]">{plan.description}</CardDescription>
                  <div className="pt-4">
                    <span className="text-4xl font-extrabold text-gray-900">{plan.price}</span>
                    <span className="text-gray-500 font-medium"> /{plan.period}</span>
                  </div>
                </CardHeader>
                
                <CardContent className="flex-grow">
                  <ul className="space-y-4">
                    {plan.features.map((feature, index) => (
                      <li key={index} className="flex items-start gap-3">
                        <Check className="h-5 w-5 text-green-500 flex-shrink-0 mt-0.5" />
                        <span className="text-gray-700 text-sm">{feature}</span>
                      </li>
                    ))}
                  </ul>
                </CardContent>

                <div className="p-6 pt-0 mt-auto">
                  <Link href="/downloads" className="block">
                    <Button 
                      className={`w-full h-12 font-semibold ${
                        plan.highlight 
                          ? 'bg-purple-600 hover:bg-purple-700 text-white' 
                          : 'bg-white border-2 border-gray-200 text-gray-900 hover:bg-gray-50'
                      }`}
                    >
                      {plan.highlight ? 'Start IRD Trial' : 'Start Free Trial'}
                    </Button>
                  </Link>
                </div>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Ideal For */}
      <section className="py-16 bg-white">
        <div className="container mx-auto px-4">
          <div className="max-w-3xl mx-auto text-center">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Ideal For
            </h2>
            <p className="text-lg text-gray-600 mb-8">
              Perfect solution for businesses with mobile or remote teams
            </p>
          </div>
          <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-4 max-w-5xl mx-auto">
            {['Remote teams', 'Multiple locations', 'Field sales', 'IRD/VAT Registered'].map((item, index) => (
              <Card key={index} className="border-2 text-center">
                <CardContent className="p-6">
                  <Cloud className="h-10 w-10 text-purple-600 mx-auto mb-3" />
                  <h3 className="font-semibold text-gray-900">{item}</h3>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* CTA */}
      <section className="py-16 bg-gradient-to-r from-purple-500 to-purple-600 text-white">
        <div className="container mx-auto px-4 text-center">
          <h2 className="text-3xl lg:text-4xl font-bold mb-4">
            Experience the Future of Accounting
          </h2>
          <p className="text-lg text-purple-100 mb-8 max-w-2xl mx-auto">
            Sign up for Busy on Cloud today and enjoy freedom of cloud-based accounting.
          </p>
          <Link href="/downloads">
            <Button size="lg" variant="outline" className="border-white text-white hover:bg-white hover:text-purple-600 px-8">
              Start Free Trial
            </Button>
          </Link>
        </div>
      </section>
    </>
  );
}