import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardFooter, CardTitle, CardDescription } from '@/components/ui/card';
import Image from 'next/image';
import {
  CheckCircle2,
  BarChart3,
  Download,
  Users,
  Shield,
  Zap,
  FileText,
  Database,
  TrendingUp,
  Clock,
  Phone,
  XCircle,
  Check,
} from 'lucide-react';
import Link from 'next/link';
import { Badge } from '@/components/ui/badge';

export default function Busy21BasicPage() {
  const features = [
    { icon: BarChart3, title: 'Complete Accounting', description: 'Manage all your accounting needs with ease' },
    { icon: Database, title: 'Inventory Management', description: 'Track stock, manage purchases and sales' },
    { icon: Shield, title: 'VAT Compliance', description: 'Stay compliant with Nepal tax regulations' },
    { icon: FileText, title: '50+ Reports', description: 'Generate reports for informed decisions' },
    { icon: Zap, title: 'Fast & Efficient', description: 'Optimized for performance on standard hardware' },
  ];

  const capabilities = [
    {
      title: 'Accounting',
      items: ['Journal Entries', 'Ledger Management', 'Trial Balance', 'Balance Sheet', 'Profit & Loss'],
    },
    {
      title: 'Inventory',
      items: ['Stock Management', 'Purchase Orders', 'Sales Orders', 'Stock Transfer', 'Batch/Serial Tracking'],
    },
    {
      title: 'Internal Accounting',
      items: ['Invoicing', 'Credit/Debit Notes', 'Receipts', 'Payments', 'Multiple Currency'],
    },
    {
      title: 'Reports',
      items: ['Financial Reports', 'Inventory Reports', 'Tax Reports', 'Customer Aging', 'Supplier Aging'],
    },
  ];

  const plans = [
    {
      name: 'Basic Single User',
      description: 'Ideal for freelancers and individual proprietors.',
      ctaText: 'Start Trial',
      highlight: true,
      badge: 'Most Popular',
      pricingOptions: [
        { period: 'Per Year', price: 'NPR 10,000' },
        { period: '3 Years', price: 'NPR 20,000' },
      ],
      features: [
        'Single User Access',
        'Complete Accounting Module',
        'Inventory Management',
        'GST/VAT Compliance',
        '50+ Reports',
        '1 Year Technical Support',
        'Software Updates (1 Year)',
      ],
    },
    {
      name: 'Basic Multi User',
      description: 'Best for small teams requiring network access.',
      ctaText: 'Start Trial',
      highlight: false,
      pricingOptions: [
        { period: 'Per Year', price: 'NPR 20,000' },
        { period: '3 Years', price: 'NPR 42,000' },
      ],
      features: [
        '5 Concurrent Users',
        'Complete Accounting Module',
        'Inventory Management',
        'GST/VAT Compliance',
        '50+ Reports',
        '1 Year Technical Support',
        'Software Updates (1 Year)',
        'Data Security Controls',
      ],
    },
  ];

  const comparisonData = [
    { feature: 'Accounting Features', single: 'Full Access', multi: 'Full Access' },
    { feature: 'Inventory Management', single: 'Full Access', multi: 'Full Access' },
    { feature: 'Nepal VAT Report', single: 'Yes', multi: 'Yes' },
    { feature: 'User Licenses', single: '1 User', multi: '5 Users (Concurrent)' },
    { feature: 'Remote Access / Network', single: 'No', multi: 'Yes (LAN)' },
    { feature: 'Role-Based Access', single: 'No', multi: 'Yes' },
    { feature: 'Data Backup Automation', single: 'Manual', multi: 'Automated/Scheduled' },
    { feature: 'Technical Support', single: '1 Year', multi: '1 Year (Priority)' },
  ];

  return (
    <>
      {/* Hero Section */}
      <section className="bg-gradient-to-br from-blue-600 to-indigo-700 text-white py-20 pt-50">
        <div className="container mx-auto px-4">
          <div className="grid lg:grid-cols-2 gap-8 items-center">
            {/* Content */}
            <div>
              <Badge className="bg-white/20 hover:bg-white/20 text-white border-white/30 mb-6">
                Version 21.0 Released
              </Badge>
              <h1 className="text-4xl lg:text-6xl font-bold mb-6 tracking-tight">
                Busy 21 Basic
              </h1>
              <p className="text-xl text-blue-100 mb-8 leading-relaxed">
                Powerful accounting software designed for growing businesses. 
                Choose the plan that fits your team size.
              </p>
              <div className="flex flex-wrap gap-4">
                <Link href="/downloads">
                  <Button size="lg" className="bg-white text-blue-700 hover:bg-blue-50 font-semibold px-8">
                    <Download className="mr-2 h-5 w-5" />
                    Download Free Trial
                  </Button>
                </Link>
                <Link href="/contact">
                  <Button size="lg" variant="outline" className="border-white text-white hover:bg-white hover:text-blue-700 px-8">
                    <Phone className="mr-2 h-5 w-5" />
                    Talk to Sales
                  </Button>
                </Link>
              </div>
            </div>
            
            {/* Image */}
            <div className="flex items-center justify-center">
              <div className="relative w-full max-w-md aspect-square bg-white/10 rounded-2xl border-2 border-white/20 overflow-hidden">
                <Image
                  src="/busy-post4.png"
                  alt="Busy 21 Basic Product"
                  fill
                  className="object-cover"
                  priority
                />
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Features Grid */}
      <section className="py-16 bg-white">
        <div className="container mx-auto px-4">
          <div className="text-center mb-16">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Essential Business Features
            </h2>
            <p className="text-lg text-gray-600 max-w-2xl mx-auto">
              Basic features are included in both Single and Multi User versions
            </p>
          </div>
          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
            {features.map((feature, index) => (
              <div key={index} className="p-6 rounded-2xl bg-gray-50 hover:bg-white hover:shadow-xl border border-transparent hover:border-gray-100 transition-all duration-300">
                <div className="w-14 h-14 bg-blue-100 rounded-2xl flex items-center justify-center mb-5">
                  <feature.icon className="h-7 w-7 text-blue-600" />
                </div>
                <h3 className="text-xl font-bold text-gray-900 mb-3">
                  {feature.title}
                </h3>
                <p className="text-gray-600 leading-relaxed">
                  {feature.description}
                </p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Capabilities */}
      <section className="py-16 bg-gray-50 border-y border-gray-100">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Deep Functionality
            </h2>
            <p className="text-lg text-gray-600 max-w-2xl mx-auto">
              Comprehensive modules to cover all business aspects
            </p>
          </div>
          <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-6">
            {capabilities.map((capability, index) => (
              <Card key={index} className="border-none shadow-sm bg-white">
                <CardHeader>
                  <h3 className="text-lg font-bold text-gray-900">
                    {capability.title}
                  </h3>
                </CardHeader>
                <CardContent>
                  <ul className="space-y-3">
                    {capability.items.map((item, i) => (
                      <li key={i} className="flex items-center gap-3 text-sm text-gray-600">
                        <CheckCircle2 className="h-4 w-4 text-blue-500 flex-shrink-0" />
                        {item}
                      </li>
                    ))}
                  </ul>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Pricing Section */}
      <section id="pricing" className="py-20 bg-white relative overflow-hidden">
        <div className="container mx-auto px-4 relative z-10">
          <div className="text-center mb-16">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Simple, Transparent Pricing
            </h2>
            <p className="text-lg text-gray-600">
              Annual or 3 years subscription. Includes 1 year of support and updates.
            </p>
          </div>
          
          <div className="grid md:grid-cols-2 gap-8 max-w-5xl mx-auto">
            {plans.map((plan, index) => (
              <Card 
                key={index} 
                className={`relative overflow-hidden flex flex-col ${plan.highlight ? 'border-blue-500 shadow-2xl scale-105 z-20' : 'border-gray-200 shadow-sm'}`}
              >
                {plan.highlight && (
                  <div className="absolute top-0 right-0 bg-blue-500 text-white text-xs font-bold px-3 py-1 rounded-bl-lg">
                    {plan.badge}
                  </div>
                )}
                <CardHeader className="pb-6">
                  <CardTitle className="text-2xl">{plan.name}</CardTitle>
                  <CardDescription className="text-base pt-2">{plan.description}</CardDescription>
                  <div className="pt-6 space-y-3">
                    {plan.pricingOptions.map((option, idx) => (
                      <div key={idx} className="flex items-baseline justify-between pb-3 border-b border-gray-200 last:border-0">
                        <span className="text-gray-600 font-medium">{option.period}</span>
                        <span className="text-2xl font-extrabold text-gray-900">{option.price}</span>
                      </div>
                    ))}
                  </div>
                </CardHeader>
                <CardContent className="flex-grow">
                  <ul className="space-y-4 mb-8">
                    {plan.features.map((feature, i) => (
                      <li key={i} className="flex items-start gap-3">
                        <Check className="h-5 w-5 text-green-500 flex-shrink-0 mt-0.5" />
                        <span className="text-gray-600">{feature}</span>
                      </li>
                    ))}
                  </ul>
                </CardContent>
                <CardFooter className="pt-0">
                  <Link href="/downloads" className="w-full">
                    <Button 
                      className={`w-full h-12 text-lg font-semibold ${plan.highlight ? 'bg-blue-600 hover:bg-blue-700 text-white' : 'bg-gray-100 hover:bg-gray-200 text-gray-900'}`}
                    >
                      {plan.ctaText}
                    </Button>
                  </Link>
                </CardFooter>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Comparison Table */}
      <section className="py-16 bg-gray-50">
        <div className="container mx-auto px-4">
          <div className="max-w-4xl mx-auto">
            <div className="text-center mb-12">
              <h2 className="text-3xl font-bold text-gray-900 mb-4">
                Compare Versions
              </h2>
              <p className="text-gray-600">See the differences between Single User and Multi User editions.</p>
            </div>

            <div className="bg-white rounded-xl shadow-lg border border-gray-200 overflow-hidden">
              <table className="w-full text-left border-collapse">
                <thead>
                  <tr className="bg-gray-100 border-b border-gray-200">
                    <th className="p-6 font-semibold text-gray-900 w-1/3">Feature</th>
                    <th className="p-6 font-semibold text-gray-900 text-center w-1/3">Single User</th>
                    <th className="p-6 font-semibold text-blue-600 text-center w-1/3 bg-blue-50/50">Multi User</th>
                  </tr>
                </thead>
                <tbody className="divide-y divide-gray-100">
                  {comparisonData.map((row, i) => (
                    <tr key={i} className="hover:bg-gray-50/50 transition-colors">
                      <td className="p-6 text-gray-700 font-medium">{row.feature}</td>
                      <td className="p-6 text-center text-gray-600">{row.single}</td>
                      <td className="p-6 text-center font-medium text-gray-900 bg-blue-50/30">{row.multi}</td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
            
            <div className="mt-8 text-center">
               <Link href="/contact">
                <Button className="px-6 bg-blue-700 text-white hover:bg-blue-800">
                  Need help deciding? Contact Us
                </Button>
               </Link>
            </div>
          </div>
        </div>
      </section>

      {/* Benefits */}
      <section className="py-16 bg-gradient-to-br from-gray-900 to-gray-800 text-white">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold mb-4">
              Why Choose Busy 21?
            </h2>
          </div>
          <div className="grid md:grid-cols-3 gap-8 max-w-5xl mx-auto">
            <div className="text-center">
              <Clock className="h-12 w-12 mx-auto mb-4 text-blue-400" />
              <h3 className="text-xl font-semibold mb-2">Quick Implementation</h3>
              <p className="text-gray-400 px-4">
                Get started in minutes with our easy installation and setup wizard.
              </p>
            </div>
            <div className="text-center">
              <Zap className="h-12 w-12 mx-auto mb-4 text-yellow-400" />
              <h3 className="text-xl font-semibold mb-2">User Friendly</h3>
              <p className="text-gray-400 px-4">
                Intuitive interface designed for users with no accounting background.
              </p>
            </div>
            <div className="text-center">
              <TrendingUp className="h-12 w-12 mx-auto mb-4 text-green-400" />
              <h3 className="text-xl font-semibold mb-2">Growth Ready</h3>
              <p className="text-gray-400 px-4">
                Seamlessly upgrade to Professional or Enterprise editions as you scale.
              </p>
            </div>
          </div>
        </div>
      </section>

      {/* CTA */}
      <section className="py-20 bg-gradient-to-r from-orange-600 to-orange-500 text-white">
        <div className="container mx-auto px-4 text-center">
          <h2 className="text-3xl lg:text-4xl font-bold mb-6">
            Ready to streamline your business?
          </h2>
          <p className="text-lg text-orange-100 mb-10 max-w-2xl mx-auto">
            Join thousands of businesses in Nepal using Busy 21 for their accounting needs.
          </p>
          <div className="flex flex-wrap justify-center gap-4">
            <Link href="/downloads">
              <Button size="lg" variant="outline" className="border-white text-white hover:bg-white hover:text-orange-600 px-8 h-12 text-lg">
                Download Free Trial
              </Button>
            </Link>
            <Link href="/contact">
              <Button size="lg" className="bg-white text-orange-600 hover:bg-orange-50 px-8 h-12 text-lg font-semibold">
                Get a Quote
              </Button>
            </Link>
          </div>
        </div>
      </section>
    </>
  );
}