'use client';

import { useState } from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { MapPin, Phone, Mail, Clock, Send, Loader2, CheckCircle2, AlertCircle } from 'lucide-react';

const Contact = () => {
  const [formData, setFormData] = useState({
    fullName: '',
    email: '',
    phone: '',
    company: '',
    subject: '',
    message: '',
  });

  const [errors, setErrors] = useState<Record<string, string>>({});
  const [loading, setLoading] = useState(false);
  const [success, setSuccess] = useState(false);

  const validateForm = () => {
    const newErrors: Record<string, string> = {};

    if (!formData.fullName.trim()) {
      newErrors.fullName = 'Full name is required';
    }

    if (!formData.email.trim()) {
      newErrors.email = 'Email is required';
    } else if (!/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(formData.email)) {
      newErrors.email = 'Please enter a valid email';
    }

    if (!formData.subject.trim()) {
      newErrors.subject = 'Subject is required';
    }

    if (!formData.message.trim()) {
      newErrors.message = 'Message is required';
    }

    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    if (!validateForm()) return;

    setLoading(true);

    try {
      const response = await fetch('/api/contact', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(formData),
      });

      if (response.ok) {
        setSuccess(true);
        setTimeout(() => {
          setSuccess(false);
          setFormData({
            fullName: '',
            email: '',
            phone: '',
            company: '',
            subject: '',
            message: '',
          });
        }, 3000);
      } else {
        const error = await response.json();
        setErrors({ submit: error.message || 'Message could not be sent. Please try again.' });
      }
    } catch (error) {
      setErrors({ submit: 'An error occurred. Please try again.' });
    } finally {
      setLoading(false);
    }
  };

  const contactInfo = [
    {
      icon: <MapPin className="w-6 h-6" />,
      title: 'Address',
      content: 'Head-Office: Nepalgunj, Banke, Nepal',
    },
    {
      icon: <MapPin className="w-6 h-6" />,
      title: 'Address',
      content: 'Branch-Office: Ghorahi, Dang, Nepal',
    },
    {
      icon: <Phone className="w-6 h-6" />,
      title: 'Phone',
      content: '+977-9857834143',
      href: 'tel:+9779857834143',
    },
    {
      icon: <Phone className="w-6 h-6" />,
      title: 'Phone',
      content: '+977-9840632000',
      href: 'tel:+9779840632000',
    },
    {
      icon: <Mail className="w-6 h-6" />,
      title: 'Email',
      content: 'info@busynepal.com',
      href: 'mailto:info@busynepal.com',
    },
    {
      icon: <Clock className="w-6 h-6" />,
      title: 'Business Hours',
      content: 'Sun - Fri: 10:00 AM - 6:00 PM',
    },
  ];

  const quickLinks = [
    { title: 'Get a Quote', description: 'Request a customized quote for your business' },
    { title: 'Schedule Demo', description: 'Book a free demo with our experts' },
    { title: 'Technical Support', description: 'Get help with installation or issues' },
    { title: 'Training Inquiry', description: 'Learn about our training programs' },
  ];

  return (
    <>
      {/* Header Section */}
      <section className="pt-32 pb-20 gradient-hero">
        <div className="container-custom">
          <div className="text-center max-w-3xl mx-auto">
            <span className="badge badge-blue mb-4 inline-block">Contact Us</span>
            <h1 className="text-4xl md:text-5xl font-bold mb-6">
              Get in{' '}
              <span className="text-gradient-primary">Touch</span>
            </h1>
            <p className="text-lg text-gray-600 leading-relaxed">
              Have questions about Busy accounting software? Our team is here to help.
              Reach out to us and we'll get back to you as soon as possible.
            </p>
          </div>
        </div>
      </section>

      {/* Contact Info */}
      <section className="py-20 bg-white">
        <div className="container-custom">
          <div className="grid md:grid-cols-4 gap-6 mb-16">
            {contactInfo.map((info, index) => (
              <Card key={index} className="card-hover shadow-brand border-2 hover:border-[#005a9f] transition-all duration-300">
                <CardContent className="p-6">
                  <div className="gradient-primary p-3 rounded-lg text-white w-fit mb-4">
                    {info.icon}
                  </div>
                  <h3 className="font-semibold mb-2">{info.title}</h3>
                  {info.href ? (
                    <a
                      href={info.href}
                      className="text-gray-600 hover:text-[#005a9f] transition-colors"
                    >
                      {info.content}
                    </a>
                  ) : (
                    <p className="text-gray-600">{info.content}</p>
                  )}
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Contact Form */}
      <section className="py-20 gradient-hero">
        <div className="container-custom">
          <div className="grid lg:grid-cols-2 gap-12">
            {/* Quick Links */}
            <div>
              <h2 className="text-3xl font-bold mb-6">Quick Links</h2>
              <p className="text-gray-600 mb-8">
                Choose the option that best fits your needs and we'll help you quickly.
              </p>

              <div className="space-y-4">
                {quickLinks.map((link, index) => (
                  <Card key={index} className="card-hover shadow-brand border-2 hover:border-[#005a9f] transition-all duration-300 cursor-pointer">
                    <CardContent className="p-6">
                      <h3 className="font-semibold text-lg mb-2">{link.title}</h3>
                      <p className="text-gray-600">{link.description}</p>
                    </CardContent>
                  </Card>
                ))}
              </div>
            </div>

            {/* Form */}
            <Card className="shadow-brand-xl">
              <CardHeader>
                <CardTitle>Send Us a Message</CardTitle>
                <CardDescription>
                  Fill out the form below and we'll get back to you within 24 hours.
                </CardDescription>
              </CardHeader>
              <CardContent>
                {!success ? (
                  <form onSubmit={handleSubmit} className="space-y-4">
                    <div className="grid sm:grid-cols-2 gap-4">
                      <div>
                        <Label htmlFor="fullName">Full Name *</Label>
                        <Input
                          id="fullName"
                          value={formData.fullName}
                          onChange={(e) => setFormData({ ...formData, fullName: e.target.value })}
                          placeholder="Your full name"
                          className={errors.fullName ? 'border-red-500' : ''}
                        />
                        {errors.fullName && (
                          <p className="text-sm text-red-500 mt-1">{errors.fullName}</p>
                        )}
                      </div>

                      <div>
                        <Label htmlFor="email">Email *</Label>
                        <Input
                          id="email"
                          type="email"
                          value={formData.email}
                          onChange={(e) => setFormData({ ...formData, email: e.target.value })}
                          placeholder="your@email.com"
                          className={errors.email ? 'border-red-500' : ''}
                        />
                        {errors.email && (
                          <p className="text-sm text-red-500 mt-1">{errors.email}</p>
                        )}
                      </div>
                    </div>

                    <div className="grid sm:grid-cols-2 gap-4">
                      <div>
                        <Label htmlFor="phone">Phone</Label>
                        <Input
                          id="phone"
                          value={formData.phone}
                          onChange={(e) => setFormData({ ...formData, phone: e.target.value })}
                          placeholder="98XXXXXXXX"
                        />
                      </div>

                      <div>
                        <Label htmlFor="company">Company</Label>
                        <Input
                          id="company"
                          value={formData.company}
                          onChange={(e) => setFormData({ ...formData, company: e.target.value })}
                          placeholder="Your company name"
                        />
                      </div>
                    </div>

                    <div>
                      <Label htmlFor="subject">Subject *</Label>
                      <Input
                        id="subject"
                        value={formData.subject}
                        onChange={(e) => setFormData({ ...formData, subject: e.target.value })}
                        placeholder="How can we help you?"
                        className={errors.subject ? 'border-red-500' : ''}
                      />
                      {errors.subject && (
                        <p className="text-sm text-red-500 mt-1">{errors.subject}</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="message">Message *</Label>
                      <Textarea
                        id="message"
                        value={formData.message}
                        onChange={(e) => setFormData({ ...formData, message: e.target.value })}
                        placeholder="Tell us more about your requirements..."
                        rows={6}
                        className={errors.message ? 'border-red-500' : ''}
                      />
                      {errors.message && (
                        <p className="text-sm text-red-500 mt-1">{errors.message}</p>
                      )}
                    </div>

                    {errors.submit && (
                      <div className="flex items-center gap-2 text-red-500 bg-red-50 p-3 rounded-lg">
                        <AlertCircle className="w-5 h-5" />
                        <p className="text-sm">{errors.submit}</p>
                      </div>
                    )}

                    <Button
                      type="submit"
                      disabled={loading}
                      className="w-full gradient-primary text-white py-6 font-medium"
                    >
                      {loading ? (
                        <>
                          <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                          Sending...
                        </>
                      ) : (
                        <>
                          Send Message
                          <Send className="w-4 h-4 ml-2" />
                        </>
                      )}
                    </Button>
                  </form>
                ) : (
                  <div className="text-center py-12 animate-fade-in">
                    <CheckCircle2 className="w-20 h-20 text-green-500 mx-auto mb-4" />
                    <h3 className="text-2xl font-bold mb-2">Message Sent Successfully!</h3>
                    <p className="text-gray-600">
                      Thank you for contacting us. We'll get back to you within 24 hours.
                    </p>
                  </div>
                )}
              </CardContent>
            </Card>
          </div>
        </div>
      </section>

      {/* Map Section */}
      <section className="py-20 bg-white">
        <div className="container-custom">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold mb-4">Find Us</h2>
            <p className="text-gray-600">Visit our office in Ghorahi, Dang, Nepal</p>
          </div>

          <Card className="shadow-brand-xl overflow-hidden">
            <CardContent className="p-0">
              <div className="w-full h-96">
                <iframe
                  src="https://www.google.com/maps/embed?pb=!1m18!1m12!1m3!1d3521.5133250393324!2d82.4826867760419!3d28.039346975993833!2m3!1f0!2f0!3f0!3m2!1i1024!2i768!4f13.1!3m3!1m2!1s0x399795004391533d%3A0x1eccc3c7baf88b25!2sNeha%20Infotech!5e0!3m2!1sen!2snp!4v1769169581312!5m2!1sen!2snp"
                  width="100%"
                  height="100%"
                  className="border-0"
                  allowFullScreen
                  loading="lazy"
                  referrerPolicy="no-referrer-when-downgrade"
                  title="Neha Infotech Location"
                />
              </div>
            </CardContent>
          </Card>
        </div>
      </section>
    </>
  );
};

export default Contact;
