import { NextResponse } from 'next/server';
import { db } from '@/lib/db';
import { sendEmail } from '@/lib/email';

export async function POST(request: Request) {
  try {
    const body = await request.json();
    const { fullName, email, phone, company, address, city, gstin, product } = body;

    // Validate required fields
    if (!fullName || !email || !phone || !product) {
      return NextResponse.json(
        { message: 'Missing required fields' },
        { status: 400 }
      );
    }

    // Validate email format
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(email)) {
      return NextResponse.json(
        { message: 'Invalid email format' },
        { status: 400 }
      );
    }

    // Save to database
    const registration = await db.downloadRegistration.create({
      data: {
        fullName,
        email,
        phone,
        company: company || '',
        address: address || '',
        city: city || '',
        gstin: gstin || '',
        product,
      },
    });

    // Send notification email
    const emailHtml = `
      <h2>New Download Registration Received</h2>
      <p><strong>Name:</strong> ${fullName}</p>
      <p><strong>Email:</strong> ${email}</p>
      <p><strong>Phone:</strong> ${phone}</p>
      ${company ? `<p><strong>Company:</strong> ${company}</p>` : ''}
      ${address ? `<p><strong>Address:</strong> ${address}</p>` : ''}
      ${city ? `<p><strong>City:</strong> ${city}</p>` : ''}
      ${gstin ? `<p><strong>GSTIN:</strong> ${gstin}</p>` : ''}
      <p><strong>Product:</strong> ${product}</p>
      <p style="color: #666; font-size: 12px; margin-top: 20px;">
        Registered at: ${new Date().toLocaleString()}
      </p>
    `;

    await sendEmail({
      to: 'tushar.neupane@gmail.com',
      subject: `New Download Registration: ${product}`,
      html: emailHtml,
    });

    return NextResponse.json(registration, { status: 201 });
  } catch (error) {
    console.error('Download registration error:', error);
    return NextResponse.json(
      { message: 'Failed to register' },
      { status: 500 }
    );
  }
}

export async function GET() {
  try {
    const registrations = await db.downloadRegistration.findMany({
      orderBy: { createdAt: 'desc' },
    });
    return NextResponse.json(registrations);
  } catch (error) {
    console.error('Error fetching registrations:', error);
    return NextResponse.json(
      { message: 'Failed to fetch registrations' },
      { status: 500 }
    );
  }
}

export async function DELETE(request: Request) {
  try {
    const url = new URL(request.url);
    const id = url.pathname.split('/').pop();

    if (!id) {
      return NextResponse.json(
        { message: 'Registration ID is required' },
        { status: 400 }
      );
    }

    await db.downloadRegistration.delete({
      where: { id },
    });

    return NextResponse.json(
      { message: 'Registration deleted successfully' },
      { status: 200 }
    );
  } catch (error) {
    console.error('Error deleting registration:', error);
    return NextResponse.json(
      { message: 'Failed to delete registration' },
      { status: 500 }
    );
  }
}
