import { NextResponse } from 'next/server';
import { db } from '@/lib/db';

export async function POST(request: Request) {
  try {
    const body = await request.json();
    const { fullName, email, phone, company, address, city, gstin, product } = body;

    // Validate required fields
    if (!fullName || !email || !phone || !product) {
      return NextResponse.json(
        { message: 'Missing required fields' },
        { status: 400 }
      );
    }

    // Validate email format
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(email)) {
      return NextResponse.json(
        { message: 'Invalid email format' },
        { status: 400 }
      );
    }

    // Validate phone (should be 10 digits)
    const cleanPhone = phone.replace(/[\s-]/g, '');
    if (!/^[0-9]{10}$/.test(cleanPhone)) {
      return NextResponse.json(
        { message: 'Phone must be 10 digits' },
        { status: 400 }
      );
    }

    // Check if email already registered for this product
    const existingRegistration = await db.downloadRegistration.findFirst({
      where: {
        email,
        product,
      },
    });

    if (existingRegistration) {
      return NextResponse.json(
        { message: 'You have already registered for this product. Please download directly.' },
        { status: 409 }
      );
    }

    // Save to database
    await db.downloadRegistration.create({
      data: {
        fullName,
        email,
        phone: cleanPhone,
        company: company || '',
        address: address || '',
        city: city || '',
        gstin: gstin || '',
        product,
      },
    });

    return NextResponse.json(
      { message: 'Registration successful' },
      { status: 200 }
    );
  } catch (error) {
    console.error('Download registration error:', error);
    return NextResponse.json(
      { message: 'Registration failed' },
      { status: 500 }
    );
  }
}
