import { NextResponse } from 'next/server';
import { userStore } from '@/lib/userStore';

export async function GET() {
  try {
    // Return users without passwords
    const safeUsers = userStore.getAll().map(({ password, ...user }) => user);
    return NextResponse.json(safeUsers);
  } catch (error) {
    console.error('Error fetching users:', error);
    return NextResponse.json({ error: 'Failed to fetch users' }, { status: 500 });
  }
}

export async function POST(request: Request) {
  try {
    const body = await request.json();
    const { username, email, password, role } = body;

    // Validate input
    if (!username || !email || !password || !role) {
      return NextResponse.json(
        { error: 'Missing required fields' },
        { status: 400 }
      );
    }

    // Check if user already exists
    if (userStore.exists(username, email)) {
      return NextResponse.json(
        { error: 'Username or email already exists' },
        { status: 409 }
      );
    }

    // Create new user
    const newUser = userStore.create({
      username,
      email,
      password,
      role: role as 'admin' | 'editor' | 'viewer',
    });

    // Return user without password
    const { password: _, ...safeUser } = newUser;
    return NextResponse.json(safeUser, { status: 201 });
  } catch (error) {
    console.error('Error creating user:', error);
    return NextResponse.json(
      { error: 'Failed to create user' },
      { status: 500 }
    );
  }
}

export async function PUT(request: Request) {
  try {
    const body = await request.json();
    const { id, username, email, password, role } = body;

    // Find user
    const user = userStore.findById(id);
    if (!user) {
      return NextResponse.json({ error: 'User not found' }, { status: 404 });
    }

    // Check if username/email is taken by another user
    if (userStore.exists(username, email, id)) {
      return NextResponse.json(
        { error: 'Username or email already exists' },
        { status: 409 }
      );
    }

    // Update user
    const updated = userStore.update(id, {
      username,
      email,
      role: role as 'admin' | 'editor' | 'viewer',
      ...(password && { password }), // Only update password if provided
    });

    if (!updated) {
      return NextResponse.json({ error: 'Failed to update user' }, { status: 500 });
    }

    // Return user without password
    const { password: _, ...safeUser } = updated;
    return NextResponse.json(safeUser);
  } catch (error) {
    console.error('Error updating user:', error);
    return NextResponse.json(
      { error: 'Failed to update user' },
      { status: 500 }
    );
  }
}
