'use client';

import { useEffect, useState } from 'react';
import { useRouter } from 'next/navigation';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Mail, Phone, Building2, Trash2, Eye, Loader2, LogOut, MessageSquare, Download, FileText, BookOpen, Users, Plus, Edit, X } from 'lucide-react';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { PageCreateModal } from '@/components/admin/PageCreateModal';
import { BlogPostModal } from '@/components/admin/BlogPostModal';
import { BlogTab } from '@/components/admin/BlogTab';
import { AllPagesView } from '@/components/admin/AllPagesView';

interface ContactMessage {
  id: string;
  fullName: string;
  email: string;
  phone?: string;
  company?: string;
  subject: string;
  message: string;
  createdAt: string;
}

interface DownloadRegistration {
  id: string;
  fullName: string;
  email: string;
  phone: string;
  company?: string;
  address?: string;
  city?: string;
  gstin?: string;
  product: string;
  createdAt: string;
  updatedAt: string;
}

interface Page {
  id: string;
  title: string;
  slug: string;
  content: string;
  createdAt: string;
  editorType: 'visual' | 'code' | 'builder';
}

interface BlogPost {
  id: string;
  title: string;
  slug: string;
  excerpt: string;
  content: string;
  category: string;
  createdAt: string;
  editorType: 'visual' | 'code';
}

interface User {
  id: string;
  username: string;
  email: string;
  role: 'admin' | 'editor' | 'viewer';
  createdAt: string;
}

const AdminDashboard = () => {
  const router = useRouter();
  const [contactMessages, setContactMessages] = useState<ContactMessage[]>([]);
  const [downloads, setDownloads] = useState<DownloadRegistration[]>([]);
  const [pages, setPages] = useState<Page[]>([]);
  const [blogPosts, setBlogPosts] = useState<BlogPost[]>([]);
  const [users, setUsers] = useState<User[]>([]);
  const [loading, setLoading] = useState(true);
  const [selectedContact, setSelectedContact] = useState<ContactMessage | null>(null);
  const [selectedDownload, setSelectedDownload] = useState<DownloadRegistration | null>(null);
  const [deleting, setDeleting] = useState<string | null>(null);
  const [pageModalOpen, setPageModalOpen] = useState(false);
  const [blogModalOpen, setBlogModalOpen] = useState(false);
  const [editingBlogPost, setEditingBlogPost] = useState<BlogPost | null>(null);
  const [editingPage, setEditingPage] = useState<Page | null>(null);
  const [showUserModal, setShowUserModal] = useState(false);
  const [editingUser, setEditingUser] = useState<User | null>(null);
  const [newUserData, setNewUserData] = useState<{
    username: string;
    email: string;
    password: string;
    role: 'admin' | 'editor' | 'viewer';
  }>({
    username: '',
    email: '',
    password: '',
    role: 'editor',
  });

  useEffect(() => {
    fetchData();
  }, []);

  const fetchData = async () => {
    try {
      const [contactRes, downloadsRes, pagesRes, blogRes, usersRes] = await Promise.all([
        fetch('/api/contact'),
        fetch('/api/downloads'),
        fetch('/api/admin/pages'),
        fetch('/api/admin/blog'),
        fetch('/api/admin/users'),
      ]);

      if (contactRes.ok) {
        setContactMessages(await contactRes.json());
      }
      if (downloadsRes.ok) {
        setDownloads(await downloadsRes.json());
      }
      if (pagesRes.ok) {
        setPages(await pagesRes.json());
      }
      if (blogRes.ok) {
        setBlogPosts(await blogRes.json());
      }
      if (usersRes.ok) {
        setUsers(await usersRes.json());
      }
    } catch (error) {
      console.error('Error fetching data:', error);
    } finally {
      setLoading(false);
    }
  };

  const deleteContact = async (id: string) => {
    if (!confirm('Are you sure you want to delete this message?')) return;

    setDeleting(id);
    try {
      const response = await fetch(`/api/contact/${id}`, {
        method: 'DELETE',
      });

      if (response.ok) {
        setContactMessages(contactMessages.filter((msg) => msg.id !== id));
        setSelectedContact(null);
      }
    } catch (error) {
      console.error('Error deleting message:', error);
    } finally {
      setDeleting(null);
    }
  };

  const deleteDownload = async (id: string) => {
    if (!confirm('Are you sure you want to delete this registration?')) return;

    setDeleting(id);
    try {
      const response = await fetch(`/api/downloads/${id}`, {
        method: 'DELETE',
      });

      if (response.ok) {
        setDownloads(downloads.filter((d) => d.id !== id));
        setSelectedDownload(null);
      }
    } catch (error) {
      console.error('Error deleting registration:', error);
    } finally {
      setDeleting(null);
    }
  };

  const handleLogout = async () => {
    try {
      await fetch('/api/admin/logout', { method: 'POST' });
      router.push('/admin');
    } catch (error) {
      console.error('Logout error:', error);
    }
  };

  // Page Management Handlers
  const handleCreatePage = async (pageData: {
    title: string;
    slug: string;
    content: string;
    editorType: 'visual' | 'code' | 'builder';
  }) => {
    try {
      const url = editingPage ? `/api/admin/pages/${editingPage.id}` : '/api/admin/pages';
      const method = editingPage ? 'PUT' : 'POST';

      const response = await fetch(url, {
        method,
        headers: { 'Content-Type': 'application/json' },
        body: editingPage ? JSON.stringify({ ...pageData, id: editingPage.id }) : JSON.stringify(pageData),
      });

      if (response.ok) {
        const result = await response.json();
        if (editingPage) {
          setPages(pages.map((p) => (p.id === result.id ? result : p)));
          setEditingPage(null);
          alert('Page updated successfully!');
        } else {
          setPages([...pages, result]);
          alert('Page created successfully!');
        }
      } else {
        alert('Failed to save page');
      }
    } catch (error) {
      console.error('Error saving page:', error);
      alert('Error saving page');
    }
  };

  const handleDeletePage = async (pageId: string) => {
    try {
      const response = await fetch(`/api/admin/pages/${pageId}`, {
        method: 'DELETE',
      });

      if (response.ok) {
        setPages(pages.filter((p) => p.id !== pageId));
      }
    } catch (error) {
      console.error('Error deleting page:', error);
    }
  };

  const handleEditPage = (page: Page) => {
    setEditingPage(page);
    setPageModalOpen(true);
  };

  // Blog Management Handlers
  const handleCreateBlogPost = async (postData: {
    title: string;
    slug: string;
    excerpt: string;
    content: string;
    category: string;
    editorType: 'visual' | 'code';
  }) => {
    try {
      const response = await fetch('/api/admin/blog', {
        method: editingBlogPost ? 'PUT' : 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: editingBlogPost
          ? JSON.stringify({ ...postData, id: editingBlogPost.id })
          : JSON.stringify(postData),
      });

      if (response.ok) {
        const result = await response.json();
        if (editingBlogPost) {
          setBlogPosts(blogPosts.map((p) => (p.id === result.id ? result : p)));
          setEditingBlogPost(null);
          alert('Blog post updated successfully!');
        } else {
          setBlogPosts([...blogPosts, result]);
          alert('Blog post created successfully!');
        }
      } else {
        alert('Failed to save blog post');
      }
    } catch (error) {
      console.error('Error saving blog post:', error);
      alert('Error saving blog post');
    }
  };

  const handleDeleteBlogPost = async (postId: string) => {
    try {
      const response = await fetch(`/api/admin/blog/${postId}`, {
        method: 'DELETE',
      });

      if (response.ok) {
        setBlogPosts(blogPosts.filter((p) => p.id !== postId));
      }
    } catch (error) {
      console.error('Error deleting blog post:', error);
    }
  };

  const handleEditBlogPost = (post: BlogPost) => {
    setEditingBlogPost(post);
    setBlogModalOpen(true);
  };

  // User Management Handlers
  const handleCreateUser = async () => {
    if (!newUserData.username || !newUserData.email || !newUserData.password) {
      alert('Please fill in all fields');
      return;
    }

    try {
      const response = await fetch('/api/admin/users', {
        method: editingUser ? 'PUT' : 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: editingUser
          ? JSON.stringify({ ...newUserData, id: editingUser.id })
          : JSON.stringify(newUserData),
      });

      if (response.ok) {
        const result = await response.json();
        if (editingUser) {
          setUsers(users.map((u) => (u.id === result.id ? result : u)));
          alert('User updated successfully!');
        } else {
          setUsers([...users, result]);
          alert('User created successfully!');
        }
        setNewUserData({ username: '', email: '', password: '', role: 'editor' });
        setEditingUser(null);
        setShowUserModal(false);
      } else {
        alert('Failed to save user');
      }
    } catch (error) {
      console.error('Error saving user:', error);
      alert('Error saving user');
    }
  };

  const handleDeleteUser = async (userId: string) => {
    if (!confirm('Are you sure you want to delete this user?')) return;

    try {
      const response = await fetch(`/api/admin/users/${userId}`, {
        method: 'DELETE',
      });

      if (response.ok) {
        setUsers(users.filter((u) => u.id !== userId));
        alert('User deleted successfully!');
      }
    } catch (error) {
      console.error('Error deleting user:', error);
      alert('Error deleting user');
    }
  };

  const handleEditUser = (user: User) => {
    setEditingUser(user);
    setNewUserData({
      username: user.username,
      email: user.email,
      password: '',
      role: user.role,
    });
    setShowUserModal(true);
  };

  const handleCloseUserModal = () => {
    setShowUserModal(false);
    setEditingUser(null);
    setNewUserData({ username: '', email: '', password: '', role: 'editor' });
  };

  const formatDate = (date: string) => {
    return new Date(date).toLocaleString();
  };

  if (loading) {
    return (
      <div className="pt-32 pb-20 min-h-screen flex items-center justify-center">
        <div className="text-center">
          <Loader2 className="w-10 h-10 animate-spin mx-auto mb-4" />
          <p>Loading dashboard...</p>
        </div>
      </div>
    );
  }

  return (
    <div className="pt-45 pb-20 min-h-screen bg-gray-50">
      <div className="container-custom">
        {/* Header */}
        <div className="flex justify-between items-start mb-8 gap-4">
          <div className="flex-1">
            <h1 className="text-4xl font-bold mb-2">Admin Dashboard</h1>
            <p className="text-gray-600">Manage all submissions and registrations</p>
          </div>
          <Button
            onClick={handleLogout}
            variant="outline"
            className="gap-2 whitespace-nowrap flex-shrink-0"
          >
            <LogOut className="w-4 h-4" />
            Logout
          </Button>
        </div>

        {/* Stats */}
        <div className="grid md:grid-cols-5 gap-4 mb-8">
          <Card>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-600 text-sm">Contact Messages</p>
                  <p className="text-3xl font-bold">{contactMessages.length}</p>
                </div>
                <MessageSquare className="w-12 h-12 text-blue-500 opacity-20" />
              </div>
            </CardContent>
          </Card>
          <Card>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-600 text-sm">Download Registrations</p>
                  <p className="text-3xl font-bold">{downloads.length}</p>
                </div>
                <Download className="w-12 h-12 text-green-500 opacity-20" />
              </div>
            </CardContent>
          </Card>
          <Card>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-600 text-sm">Pages</p>
                  <p className="text-3xl font-bold">{pages.length}</p>
                </div>
                <FileText className="w-12 h-12 text-purple-500 opacity-20" />
              </div>
            </CardContent>
          </Card>
          <Card>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-600 text-sm">Blog Posts</p>
                  <p className="text-3xl font-bold">{blogPosts.length}</p>
                </div>
                <BookOpen className="w-12 h-12 text-orange-500 opacity-20" />
              </div>
            </CardContent>
          </Card>
          <Card>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-600 text-sm">Users</p>
                  <p className="text-3xl font-bold">{users.length}</p>
                </div>
                <Users className="w-12 h-12 text-red-500 opacity-20" />
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Tabs */}
        <Tabs defaultValue="all-pages" className="space-y-6">
          <TabsList className="grid w-full max-w-4xl grid-cols-5">
            <TabsTrigger value="all-pages" className="gap-2">
              <FileText className="w-4 h-4" />
              All Pages
            </TabsTrigger>
            <TabsTrigger value="contacts" className="gap-2">
              <MessageSquare className="w-4 h-4" />
              Messages
            </TabsTrigger>
            <TabsTrigger value="downloads" className="gap-2">
              <Download className="w-4 h-4" />
              Registrations
            </TabsTrigger>
            <TabsTrigger value="blog" className="gap-2">
              <BookOpen className="w-4 h-4" />
              Blog
            </TabsTrigger>
            <TabsTrigger value="users" className="gap-2">
              <Users className="w-4 h-4" />
              Users
            </TabsTrigger>
          </TabsList>

          {/* All Pages Tab */}
          <TabsContent value="all-pages">
            <AllPagesView
              onEditPage={handleEditPage}
              onCreatePage={() => {
                setEditingPage(null);
                setPageModalOpen(true);
              }}
            />
          </TabsContent>

          {/* Contact Messages Tab */}
          <TabsContent value="contacts">
            <div className="grid lg:grid-cols-3 gap-8">
              {/* Messages List */}
              <div className="lg:col-span-2">
                <Card>
                  <CardHeader>
                    <CardTitle>Contact Messages</CardTitle>
                    <CardDescription>Click on a message to view details</CardDescription>
                  </CardHeader>
                  <CardContent>
                    {contactMessages.length === 0 ? (
                      <p className="text-center text-gray-500 py-8">No messages yet</p>
                    ) : (
                      <div className="space-y-3">
                        {contactMessages.map((message) => (
                          <div
                            key={message.id}
                            onClick={() => setSelectedContact(message)}
                            className={`p-4 border rounded-lg cursor-pointer transition-all ${
                              selectedContact?.id === message.id
                                ? 'border-blue-500 bg-blue-50'
                                : 'border-gray-200 hover:border-gray-300'
                            }`}
                          >
                            <div className="flex items-start justify-between">
                              <div className="flex-1">
                                <h3 className="font-semibold">{message.fullName}</h3>
                                <p className="text-sm text-gray-600 truncate">{message.subject}</p>
                                <p className="text-xs text-gray-500 mt-1">
                                  {formatDate(message.createdAt)}
                                </p>
                              </div>
                              <Eye className="w-4 h-4 text-gray-400 flex-shrink-0" />
                            </div>
                          </div>
                        ))}
                      </div>
                    )}
                  </CardContent>
                </Card>
              </div>

              {/* Details Panel */}
              <div>
                {selectedContact ? (
                  <Card>
                    <CardHeader>
                      <CardTitle className="text-lg">Message Details</CardTitle>
                    </CardHeader>
                    <CardContent className="space-y-4">
                      <div>
                        <label className="text-xs font-semibold text-gray-500 uppercase">Name</label>
                        <p className="text-base font-medium">{selectedContact.fullName}</p>
                      </div>

                      <div>
                        <label className="text-xs font-semibold text-gray-500 uppercase">Email</label>
                        <a
                          href={`mailto:${selectedContact.email}`}
                          className="text-blue-600 hover:underline flex items-center gap-2"
                        >
                          <Mail className="w-4 h-4" />
                          {selectedContact.email}
                        </a>
                      </div>

                      {selectedContact.phone && (
                        <div>
                          <label className="text-xs font-semibold text-gray-500 uppercase">Phone</label>
                          <a
                            href={`tel:${selectedContact.phone}`}
                            className="text-blue-600 hover:underline flex items-center gap-2"
                          >
                            <Phone className="w-4 h-4" />
                            {selectedContact.phone}
                          </a>
                        </div>
                      )}

                      {selectedContact.company && (
                        <div>
                          <label className="text-xs font-semibold text-gray-500 uppercase">Company</label>
                          <p className="text-sm flex items-center gap-2">
                            <Building2 className="w-4 h-4" />
                            {selectedContact.company}
                          </p>
                        </div>
                      )}

                      <div>
                        <label className="text-xs font-semibold text-gray-500 uppercase">Subject</label>
                        <p className="text-sm font-medium">{selectedContact.subject}</p>
                      </div>

                      <div>
                        <label className="text-xs font-semibold text-gray-500 uppercase">Message</label>
                        <p className="text-sm text-gray-700 whitespace-pre-wrap bg-gray-50 p-3 rounded">
                          {selectedContact.message}
                        </p>
                      </div>

                      <div>
                        <label className="text-xs font-semibold text-gray-500 uppercase">Received</label>
                        <p className="text-sm text-gray-600">
                          {formatDate(selectedContact.createdAt)}
                        </p>
                      </div>

                      <Button
                        onClick={() => deleteContact(selectedContact.id)}
                        disabled={deleting === selectedContact.id}
                        variant="destructive"
                        className="w-full"
                      >
                        {deleting === selectedContact.id ? (
                          <>
                            <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                            Deleting...
                          </>
                        ) : (
                          <>
                            <Trash2 className="w-4 h-4 mr-2" />
                            Delete Message
                          </>
                        )}
                      </Button>
                    </CardContent>
                  </Card>
                ) : (
                  <Card>
                    <CardContent className="py-12">
                      <p className="text-center text-gray-500">
                        Select a message to view details
                      </p>
                    </CardContent>
                  </Card>
                )}
              </div>
            </div>
          </TabsContent>

          {/* Downloads Tab */}
          <TabsContent value="downloads">
            <div className="grid lg:grid-cols-3 gap-8">
              {/* Registrations List */}
              <div className="lg:col-span-2">
                <Card>
                  <CardHeader>
                    <CardTitle>Download Registrations</CardTitle>
                    <CardDescription>Click on a registration to view details</CardDescription>
                  </CardHeader>
                  <CardContent>
                    {downloads.length === 0 ? (
                      <p className="text-center text-gray-500 py-8">No registrations yet</p>
                    ) : (
                      <div className="space-y-3">
                        {downloads.map((download) => (
                          <div
                            key={download.id}
                            onClick={() => setSelectedDownload(download)}
                            className={`p-4 border rounded-lg cursor-pointer transition-all ${
                              selectedDownload?.id === download.id
                                ? 'border-green-500 bg-green-50'
                                : 'border-gray-200 hover:border-gray-300'
                            }`}
                          >
                            <div className="flex items-start justify-between">
                              <div className="flex-1">
                                <h3 className="font-semibold">{download.fullName}</h3>
                                <p className="text-sm text-gray-600">{download.product}</p>
                                <p className="text-xs text-gray-500 mt-1">
                                  {formatDate(download.createdAt)}
                                </p>
                              </div>
                              <Eye className="w-4 h-4 text-gray-400 flex-shrink-0" />
                            </div>
                          </div>
                        ))}
                      </div>
                    )}
                  </CardContent>
                </Card>
              </div>

              {/* Details Panel */}
              <div>
                {selectedDownload ? (
                  <Card>
                    <CardHeader>
                      <CardTitle className="text-lg">Registration Details</CardTitle>
                    </CardHeader>
                    <CardContent className="space-y-4">
                      <div>
                        <label className="text-xs font-semibold text-gray-500 uppercase">Name</label>
                        <p className="text-base font-medium">{selectedDownload.fullName}</p>
                      </div>

                      <div>
                        <label className="text-xs font-semibold text-gray-500 uppercase">Email</label>
                        <a
                          href={`mailto:${selectedDownload.email}`}
                          className="text-blue-600 hover:underline flex items-center gap-2"
                        >
                          <Mail className="w-4 h-4" />
                          {selectedDownload.email}
                        </a>
                      </div>

                      <div>
                        <label className="text-xs font-semibold text-gray-500 uppercase">Phone</label>
                        <a
                          href={`tel:${selectedDownload.phone}`}
                          className="text-blue-600 hover:underline flex items-center gap-2"
                        >
                          <Phone className="w-4 h-4" />
                          {selectedDownload.phone}
                        </a>
                      </div>

                      {selectedDownload.company && (
                        <div>
                          <label className="text-xs font-semibold text-gray-500 uppercase">Company</label>
                          <p className="text-sm flex items-center gap-2">
                            <Building2 className="w-4 h-4" />
                            {selectedDownload.company}
                          </p>
                        </div>
                      )}

                      <div>
                        <label className="text-xs font-semibold text-gray-500 uppercase">Product</label>
                        <p className="text-sm font-medium">{selectedDownload.product}</p>
                      </div>

                      {selectedDownload.address && (
                        <div>
                          <label className="text-xs font-semibold text-gray-500 uppercase">Address</label>
                          <p className="text-sm text-gray-700">{selectedDownload.address}</p>
                        </div>
                      )}

                      {selectedDownload.city && (
                        <div>
                          <label className="text-xs font-semibold text-gray-500 uppercase">City</label>
                          <p className="text-sm text-gray-700">{selectedDownload.city}</p>
                        </div>
                      )}

                      {selectedDownload.gstin && (
                        <div>
                          <label className="text-xs font-semibold text-gray-500 uppercase">GSTIN</label>
                          <p className="text-sm font-mono text-gray-700">{selectedDownload.gstin}</p>
                        </div>
                      )}

                      <div>
                        <label className="text-xs font-semibold text-gray-500 uppercase">Registered</label>
                        <p className="text-sm text-gray-600">
                          {formatDate(selectedDownload.createdAt)}
                        </p>
                      </div>

                      <Button
                        onClick={() => deleteDownload(selectedDownload.id)}
                        disabled={deleting === selectedDownload.id}
                        variant="destructive"
                        className="w-full"
                      >
                        {deleting === selectedDownload.id ? (
                          <>
                            <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                            Deleting...
                          </>
                        ) : (
                          <>
                            <Trash2 className="w-4 h-4 mr-2" />
                            Delete Registration
                          </>
                        )}
                      </Button>
                    </CardContent>
                  </Card>
                ) : (
                  <Card>
                    <CardContent className="py-12">
                      <p className="text-center text-gray-500">
                        Select a registration to view details
                      </p>
                    </CardContent>
                  </Card>
                )}
              </div>
            </div>
          </TabsContent>

          {/* Blog Tab */}
          <TabsContent value="blog">
            <BlogTab
              posts={blogPosts}
              loading={loading}
              onCreateNew={() => {
                setEditingBlogPost(null);
                setBlogModalOpen(true);
              }}
              onEdit={handleEditBlogPost}
              onDelete={handleDeleteBlogPost}
            />
          </TabsContent>

          {/* Users Tab */}
          <TabsContent value="users">
            <Card>
              <CardHeader className="flex flex-row items-center justify-between">
                <div>
                  <CardTitle>User Management</CardTitle>
                  <CardDescription>Create and manage admin users</CardDescription>
                </div>
                <Button
                  onClick={() => setShowUserModal(true)}
                  className="gap-2"
                >
                  <Plus className="w-4 h-4" />
                  Create User
                </Button>
              </CardHeader>
              <CardContent>
                {users.length === 0 ? (
                  <p className="text-center text-gray-500 py-8">No users created yet</p>
                ) : (
                  <div className="overflow-x-auto">
                    <table className="w-full">
                      <thead className="border-b">
                        <tr>
                          <th className="text-left p-3 font-semibold">Username</th>
                          <th className="text-left p-3 font-semibold">Email</th>
                          <th className="text-left p-3 font-semibold">Role</th>
                          <th className="text-left p-3 font-semibold">Created At</th>
                          <th className="text-right p-3 font-semibold">Actions</th>
                        </tr>
                      </thead>
                      <tbody>
                        {users.map((user) => (
                          <tr key={user.id} className="border-b hover:bg-gray-50">
                            <td className="p-3">{user.username}</td>
                            <td className="p-3">{user.email}</td>
                            <td className="p-3">
                              <span className={`px-3 py-1 rounded-full text-xs font-semibold ${
                                user.role === 'admin'
                                  ? 'bg-red-100 text-red-800'
                                  : user.role === 'editor'
                                  ? 'bg-blue-100 text-blue-800'
                                  : 'bg-gray-100 text-gray-800'
                              }`}>
                                {user.role}
                              </span>
                            </td>
                            <td className="p-3 text-sm text-gray-600">
                              {formatDate(user.createdAt)}
                            </td>
                            <td className="p-3 text-right">
                              <div className="flex gap-2 justify-end">
                                <Button
                                  size="sm"
                                  variant="outline"
                                  onClick={() => handleEditUser(user)}
                                  className="gap-1"
                                >
                                  <Edit className="w-4 h-4" />
                                  Edit
                                </Button>
                                <Button
                                  size="sm"
                                  variant="destructive"
                                  onClick={() => handleDeleteUser(user.id)}
                                  className="gap-1"
                                >
                                  <Trash2 className="w-4 h-4" />
                                  Delete
                                </Button>
                              </div>
                            </td>
                          </tr>
                        ))}
                      </tbody>
                    </table>
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>

        {/* User Modal */}
        {showUserModal && (
          <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50 p-4">
            <Card className="w-full max-w-md">
              <CardHeader className="flex flex-row items-center justify-between">
                <CardTitle>{editingUser ? 'Edit User' : 'Create New User'}</CardTitle>
                <Button
                  variant="ghost"
                  size="sm"
                  onClick={handleCloseUserModal}
                >
                  <X className="w-4 h-4" />
                </Button>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <Label htmlFor="username">Username</Label>
                  <Input
                    id="username"
                    value={newUserData.username}
                    onChange={(e) =>
                      setNewUserData({ ...newUserData, username: e.target.value })
                    }
                    placeholder="Enter username"
                  />
                </div>
                <div>
                  <Label htmlFor="email">Email</Label>
                  <Input
                    id="email"
                    type="email"
                    value={newUserData.email}
                    onChange={(e) =>
                      setNewUserData({ ...newUserData, email: e.target.value })
                    }
                    placeholder="Enter email"
                  />
                </div>
                <div>
                  <Label htmlFor="password">
                    Password {editingUser && <span className="text-sm text-gray-500">(leave empty to keep current)</span>}
                  </Label>
                  <Input
                    id="password"
                    type="password"
                    value={newUserData.password}
                    onChange={(e) =>
                      setNewUserData({ ...newUserData, password: e.target.value })
                    }
                    placeholder="Enter password"
                  />
                </div>
                <div>
                  <Label htmlFor="role">Role</Label>
                  <select
                    id="role"
                    aria-label="User role"
                    value={newUserData.role}
                    onChange={(e) =>
                      setNewUserData({
                        ...newUserData,
                        role: e.target.value as 'admin' | 'editor' | 'viewer',
                      })
                    }
                    className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                  >
                    <option value="viewer">Viewer</option>
                    <option value="editor">Editor</option>
                    <option value="admin">Admin</option>
                  </select>
                </div>
                <div className="flex gap-2 justify-end pt-4">
                  <Button
                    variant="outline"
                    onClick={handleCloseUserModal}
                  >
                    Cancel
                  </Button>
                  <Button onClick={handleCreateUser}>
                    {editingUser ? 'Update User' : 'Create User'}
                  </Button>
                </div>
              </CardContent>
            </Card>
          </div>
        )}
      </div>

      {/* Modals */}
      <PageCreateModal
        isOpen={pageModalOpen}
        onClose={() => {
          setPageModalOpen(false);
          setEditingPage(null);
        }}
        onCreate={handleCreatePage}
        editData={editingPage}
      />

      <BlogPostModal
        isOpen={blogModalOpen}
        onClose={() => {
          setBlogModalOpen(false);
          setEditingBlogPost(null);
        }}
        onCreate={handleCreateBlogPost}
        editData={editingBlogPost}
      />
    </div>
  );
};

export default AdminDashboard;