import { NextResponse } from 'next/server';
import type { NextRequest } from 'next/server';

export function middleware(request: NextRequest) {
  const { pathname, host } = request.nextUrl;
  
  // Handle WWW to non-WWW canonicalization
  if (host.startsWith('www.')) {
    const newUrl = request.nextUrl.clone();
    newUrl.host = host.replace(/^www\./, '');
    return NextResponse.redirect(newUrl, 301);
  }

  // Only protect /admin/dashboard and /admin/messages routes
  if (pathname.startsWith('/admin/dashboard') || pathname.startsWith('/admin/messages')) {
    const authCookie = request.cookies.get('adminAuth');
    
    if (!authCookie) {
      // Redirect to login page
      return NextResponse.redirect(new URL('/admin', request.url));
    }
  }

  const response = NextResponse.next();
  
  // Add security headers
  response.headers.set('Strict-Transport-Security', 'max-age=31536000; includeSubDomains; preload');
  response.headers.set('X-Content-Type-Options', 'nosniff');
  response.headers.set('X-Frame-Options', 'SAMEORIGIN');
  response.headers.set('X-XSS-Protection', '1; mode=block');
  response.headers.set('Referrer-Policy', 'strict-origin-when-cross-origin');

  return response;
}

export const config = {
  matcher: ['/admin/:path*', '/:path*'],
};
